/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.wafv2.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A rule statement that inspects web traffic based on the Autonomous System Number (ASN) associated with the request's
 * IP address.
 * </p>
 * <p>
 * For additional details, see <a
 * href="https://docs.aws.amazon.com/waf/latest/developerguide/waf-rule-statement-type-asn-match.html">ASN match rule
 * statement</a> in the <a href="https://docs.aws.amazon.com/waf/latest/developerguide/waf-chapter.html">WAF Developer
 * Guide</a>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class AsnMatchStatement implements SdkPojo, Serializable,
        ToCopyableBuilder<AsnMatchStatement.Builder, AsnMatchStatement> {
    private static final SdkField<List<Long>> ASN_LIST_FIELD = SdkField
            .<List<Long>> builder(MarshallingType.LIST)
            .memberName("AsnList")
            .getter(getter(AsnMatchStatement::asnList))
            .setter(setter(Builder::asnList))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AsnList").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Long> builder(MarshallingType.LONG)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<ForwardedIPConfig> FORWARDED_IP_CONFIG_FIELD = SdkField
            .<ForwardedIPConfig> builder(MarshallingType.SDK_POJO).memberName("ForwardedIPConfig")
            .getter(getter(AsnMatchStatement::forwardedIPConfig)).setter(setter(Builder::forwardedIPConfig))
            .constructor(ForwardedIPConfig::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ForwardedIPConfig").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ASN_LIST_FIELD,
            FORWARDED_IP_CONFIG_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final List<Long> asnList;

    private final ForwardedIPConfig forwardedIPConfig;

    private AsnMatchStatement(BuilderImpl builder) {
        this.asnList = builder.asnList;
        this.forwardedIPConfig = builder.forwardedIPConfig;
    }

    /**
     * For responses, this returns true if the service returned a value for the AsnList property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasAsnList() {
        return asnList != null && !(asnList instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Contains one or more Autonomous System Numbers (ASNs). ASNs are unique identifiers assigned to large internet
     * networks managed by organizations such as internet service providers, enterprises, universities, or government
     * agencies.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasAsnList} method.
     * </p>
     * 
     * @return Contains one or more Autonomous System Numbers (ASNs). ASNs are unique identifiers assigned to large
     *         internet networks managed by organizations such as internet service providers, enterprises, universities,
     *         or government agencies.
     */
    public final List<Long> asnList() {
        return asnList;
    }

    /**
     * <p>
     * The configuration for inspecting IP addresses to match against an ASN in an HTTP header that you specify, instead
     * of using the IP address that's reported by the web request origin. Commonly, this is the X-Forwarded-For (XFF)
     * header, but you can specify any header name.
     * </p>
     * 
     * @return The configuration for inspecting IP addresses to match against an ASN in an HTTP header that you specify,
     *         instead of using the IP address that's reported by the web request origin. Commonly, this is the
     *         X-Forwarded-For (XFF) header, but you can specify any header name.
     */
    public final ForwardedIPConfig forwardedIPConfig() {
        return forwardedIPConfig;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(hasAsnList() ? asnList() : null);
        hashCode = 31 * hashCode + Objects.hashCode(forwardedIPConfig());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof AsnMatchStatement)) {
            return false;
        }
        AsnMatchStatement other = (AsnMatchStatement) obj;
        return hasAsnList() == other.hasAsnList() && Objects.equals(asnList(), other.asnList())
                && Objects.equals(forwardedIPConfig(), other.forwardedIPConfig());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("AsnMatchStatement").add("AsnList", hasAsnList() ? asnList() : null)
                .add("ForwardedIPConfig", forwardedIPConfig()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "AsnList":
            return Optional.ofNullable(clazz.cast(asnList()));
        case "ForwardedIPConfig":
            return Optional.ofNullable(clazz.cast(forwardedIPConfig()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("AsnList", ASN_LIST_FIELD);
        map.put("ForwardedIPConfig", FORWARDED_IP_CONFIG_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<AsnMatchStatement, T> g) {
        return obj -> g.apply((AsnMatchStatement) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, AsnMatchStatement> {
        /**
         * <p>
         * Contains one or more Autonomous System Numbers (ASNs). ASNs are unique identifiers assigned to large internet
         * networks managed by organizations such as internet service providers, enterprises, universities, or
         * government agencies.
         * </p>
         * 
         * @param asnList
         *        Contains one or more Autonomous System Numbers (ASNs). ASNs are unique identifiers assigned to large
         *        internet networks managed by organizations such as internet service providers, enterprises,
         *        universities, or government agencies.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder asnList(Collection<Long> asnList);

        /**
         * <p>
         * Contains one or more Autonomous System Numbers (ASNs). ASNs are unique identifiers assigned to large internet
         * networks managed by organizations such as internet service providers, enterprises, universities, or
         * government agencies.
         * </p>
         * 
         * @param asnList
         *        Contains one or more Autonomous System Numbers (ASNs). ASNs are unique identifiers assigned to large
         *        internet networks managed by organizations such as internet service providers, enterprises,
         *        universities, or government agencies.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder asnList(Long... asnList);

        /**
         * <p>
         * The configuration for inspecting IP addresses to match against an ASN in an HTTP header that you specify,
         * instead of using the IP address that's reported by the web request origin. Commonly, this is the
         * X-Forwarded-For (XFF) header, but you can specify any header name.
         * </p>
         * 
         * @param forwardedIPConfig
         *        The configuration for inspecting IP addresses to match against an ASN in an HTTP header that you
         *        specify, instead of using the IP address that's reported by the web request origin. Commonly, this is
         *        the X-Forwarded-For (XFF) header, but you can specify any header name.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder forwardedIPConfig(ForwardedIPConfig forwardedIPConfig);

        /**
         * <p>
         * The configuration for inspecting IP addresses to match against an ASN in an HTTP header that you specify,
         * instead of using the IP address that's reported by the web request origin. Commonly, this is the
         * X-Forwarded-For (XFF) header, but you can specify any header name.
         * </p>
         * This is a convenience method that creates an instance of the {@link ForwardedIPConfig.Builder} avoiding the
         * need to create one manually via {@link ForwardedIPConfig#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ForwardedIPConfig.Builder#build()} is called immediately and its
         * result is passed to {@link #forwardedIPConfig(ForwardedIPConfig)}.
         * 
         * @param forwardedIPConfig
         *        a consumer that will call methods on {@link ForwardedIPConfig.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #forwardedIPConfig(ForwardedIPConfig)
         */
        default Builder forwardedIPConfig(Consumer<ForwardedIPConfig.Builder> forwardedIPConfig) {
            return forwardedIPConfig(ForwardedIPConfig.builder().applyMutation(forwardedIPConfig).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private List<Long> asnList = DefaultSdkAutoConstructList.getInstance();

        private ForwardedIPConfig forwardedIPConfig;

        private BuilderImpl() {
        }

        private BuilderImpl(AsnMatchStatement model) {
            asnList(model.asnList);
            forwardedIPConfig(model.forwardedIPConfig);
        }

        public final Collection<Long> getAsnList() {
            if (asnList instanceof SdkAutoConstructList) {
                return null;
            }
            return asnList;
        }

        public final void setAsnList(Collection<Long> asnList) {
            this.asnList = AsnListCopier.copy(asnList);
        }

        @Override
        public final Builder asnList(Collection<Long> asnList) {
            this.asnList = AsnListCopier.copy(asnList);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder asnList(Long... asnList) {
            asnList(Arrays.asList(asnList));
            return this;
        }

        public final ForwardedIPConfig.Builder getForwardedIPConfig() {
            return forwardedIPConfig != null ? forwardedIPConfig.toBuilder() : null;
        }

        public final void setForwardedIPConfig(ForwardedIPConfig.BuilderImpl forwardedIPConfig) {
            this.forwardedIPConfig = forwardedIPConfig != null ? forwardedIPConfig.build() : null;
        }

        @Override
        public final Builder forwardedIPConfig(ForwardedIPConfig forwardedIPConfig) {
            this.forwardedIPConfig = forwardedIPConfig;
            return this;
        }

        @Override
        public AsnMatchStatement build() {
            return new AsnMatchStatement(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
