/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.iam.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains information about an attached permissions boundary.
 * </p>
 * <p>
 * An attached permissions boundary is a managed policy that has been attached to a user or role to set the permissions
 * boundary.
 * </p>
 * <p>
 * For more information about permissions boundaries, see <a
 * href="https://docs.aws.amazon.com/IAM/latest/UserGuide/access_policies_boundaries.html">Permissions boundaries for
 * IAM identities </a> in the <i>IAM User Guide</i>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class AttachedPermissionsBoundary implements SdkPojo, Serializable,
        ToCopyableBuilder<AttachedPermissionsBoundary.Builder, AttachedPermissionsBoundary> {
    private static final SdkField<String> PERMISSIONS_BOUNDARY_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("PermissionsBoundaryType").getter(getter(AttachedPermissionsBoundary::permissionsBoundaryTypeAsString))
            .setter(setter(Builder::permissionsBoundaryType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PermissionsBoundaryType").build())
            .build();

    private static final SdkField<String> PERMISSIONS_BOUNDARY_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("PermissionsBoundaryArn").getter(getter(AttachedPermissionsBoundary::permissionsBoundaryArn))
            .setter(setter(Builder::permissionsBoundaryArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PermissionsBoundaryArn").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(
            PERMISSIONS_BOUNDARY_TYPE_FIELD, PERMISSIONS_BOUNDARY_ARN_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String permissionsBoundaryType;

    private final String permissionsBoundaryArn;

    private AttachedPermissionsBoundary(BuilderImpl builder) {
        this.permissionsBoundaryType = builder.permissionsBoundaryType;
        this.permissionsBoundaryArn = builder.permissionsBoundaryArn;
    }

    /**
     * <p>
     * The permissions boundary usage type that indicates what type of IAM resource is used as the permissions boundary
     * for an entity. This data type can only have a value of <code>Policy</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #permissionsBoundaryType} will return {@link PermissionsBoundaryAttachmentType#UNKNOWN_TO_SDK_VERSION}.
     * The raw value returned by the service is available from {@link #permissionsBoundaryTypeAsString}.
     * </p>
     * 
     * @return The permissions boundary usage type that indicates what type of IAM resource is used as the permissions
     *         boundary for an entity. This data type can only have a value of <code>Policy</code>.
     * @see PermissionsBoundaryAttachmentType
     */
    public final PermissionsBoundaryAttachmentType permissionsBoundaryType() {
        return PermissionsBoundaryAttachmentType.fromValue(permissionsBoundaryType);
    }

    /**
     * <p>
     * The permissions boundary usage type that indicates what type of IAM resource is used as the permissions boundary
     * for an entity. This data type can only have a value of <code>Policy</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #permissionsBoundaryType} will return {@link PermissionsBoundaryAttachmentType#UNKNOWN_TO_SDK_VERSION}.
     * The raw value returned by the service is available from {@link #permissionsBoundaryTypeAsString}.
     * </p>
     * 
     * @return The permissions boundary usage type that indicates what type of IAM resource is used as the permissions
     *         boundary for an entity. This data type can only have a value of <code>Policy</code>.
     * @see PermissionsBoundaryAttachmentType
     */
    public final String permissionsBoundaryTypeAsString() {
        return permissionsBoundaryType;
    }

    /**
     * <p>
     * The ARN of the policy used to set the permissions boundary for the user or role.
     * </p>
     * 
     * @return The ARN of the policy used to set the permissions boundary for the user or role.
     */
    public final String permissionsBoundaryArn() {
        return permissionsBoundaryArn;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(permissionsBoundaryTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(permissionsBoundaryArn());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof AttachedPermissionsBoundary)) {
            return false;
        }
        AttachedPermissionsBoundary other = (AttachedPermissionsBoundary) obj;
        return Objects.equals(permissionsBoundaryTypeAsString(), other.permissionsBoundaryTypeAsString())
                && Objects.equals(permissionsBoundaryArn(), other.permissionsBoundaryArn());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("AttachedPermissionsBoundary").add("PermissionsBoundaryType", permissionsBoundaryTypeAsString())
                .add("PermissionsBoundaryArn", permissionsBoundaryArn()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "PermissionsBoundaryType":
            return Optional.ofNullable(clazz.cast(permissionsBoundaryTypeAsString()));
        case "PermissionsBoundaryArn":
            return Optional.ofNullable(clazz.cast(permissionsBoundaryArn()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("PermissionsBoundaryType", PERMISSIONS_BOUNDARY_TYPE_FIELD);
        map.put("PermissionsBoundaryArn", PERMISSIONS_BOUNDARY_ARN_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<AttachedPermissionsBoundary, T> g) {
        return obj -> g.apply((AttachedPermissionsBoundary) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, AttachedPermissionsBoundary> {
        /**
         * <p>
         * The permissions boundary usage type that indicates what type of IAM resource is used as the permissions
         * boundary for an entity. This data type can only have a value of <code>Policy</code>.
         * </p>
         * 
         * @param permissionsBoundaryType
         *        The permissions boundary usage type that indicates what type of IAM resource is used as the
         *        permissions boundary for an entity. This data type can only have a value of <code>Policy</code>.
         * @see PermissionsBoundaryAttachmentType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see PermissionsBoundaryAttachmentType
         */
        Builder permissionsBoundaryType(String permissionsBoundaryType);

        /**
         * <p>
         * The permissions boundary usage type that indicates what type of IAM resource is used as the permissions
         * boundary for an entity. This data type can only have a value of <code>Policy</code>.
         * </p>
         * 
         * @param permissionsBoundaryType
         *        The permissions boundary usage type that indicates what type of IAM resource is used as the
         *        permissions boundary for an entity. This data type can only have a value of <code>Policy</code>.
         * @see PermissionsBoundaryAttachmentType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see PermissionsBoundaryAttachmentType
         */
        Builder permissionsBoundaryType(PermissionsBoundaryAttachmentType permissionsBoundaryType);

        /**
         * <p>
         * The ARN of the policy used to set the permissions boundary for the user or role.
         * </p>
         * 
         * @param permissionsBoundaryArn
         *        The ARN of the policy used to set the permissions boundary for the user or role.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder permissionsBoundaryArn(String permissionsBoundaryArn);
    }

    static final class BuilderImpl implements Builder {
        private String permissionsBoundaryType;

        private String permissionsBoundaryArn;

        private BuilderImpl() {
        }

        private BuilderImpl(AttachedPermissionsBoundary model) {
            permissionsBoundaryType(model.permissionsBoundaryType);
            permissionsBoundaryArn(model.permissionsBoundaryArn);
        }

        public final String getPermissionsBoundaryType() {
            return permissionsBoundaryType;
        }

        public final void setPermissionsBoundaryType(String permissionsBoundaryType) {
            this.permissionsBoundaryType = permissionsBoundaryType;
        }

        @Override
        public final Builder permissionsBoundaryType(String permissionsBoundaryType) {
            this.permissionsBoundaryType = permissionsBoundaryType;
            return this;
        }

        @Override
        public final Builder permissionsBoundaryType(PermissionsBoundaryAttachmentType permissionsBoundaryType) {
            this.permissionsBoundaryType(permissionsBoundaryType == null ? null : permissionsBoundaryType.toString());
            return this;
        }

        public final String getPermissionsBoundaryArn() {
            return permissionsBoundaryArn;
        }

        public final void setPermissionsBoundaryArn(String permissionsBoundaryArn) {
            this.permissionsBoundaryArn = permissionsBoundaryArn;
        }

        @Override
        public final Builder permissionsBoundaryArn(String permissionsBoundaryArn) {
            this.permissionsBoundaryArn = permissionsBoundaryArn;
            return this;
        }

        @Override
        public AttachedPermissionsBoundary build() {
            return new AttachedPermissionsBoundary(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
