/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.wafv2.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Configures inspection of the response header. This is part of the <code>ResponseInspection</code> configuration for
 * <code>AWSManagedRulesATPRuleSet</code> and <code>AWSManagedRulesACFPRuleSet</code>.
 * </p>
 * <note>
 * <p>
 * Response inspection is available only in web ACLs that protect Amazon CloudFront distributions.
 * </p>
 * </note>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ResponseInspectionHeader implements SdkPojo, Serializable,
        ToCopyableBuilder<ResponseInspectionHeader.Builder, ResponseInspectionHeader> {
    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Name")
            .getter(getter(ResponseInspectionHeader::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name").build()).build();

    private static final SdkField<List<String>> SUCCESS_VALUES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("SuccessValues")
            .getter(getter(ResponseInspectionHeader::successValues))
            .setter(setter(Builder::successValues))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SuccessValues").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<String>> FAILURE_VALUES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("FailureValues")
            .getter(getter(ResponseInspectionHeader::failureValues))
            .setter(setter(Builder::failureValues))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FailureValues").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAME_FIELD,
            SUCCESS_VALUES_FIELD, FAILURE_VALUES_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String name;

    private final List<String> successValues;

    private final List<String> failureValues;

    private ResponseInspectionHeader(BuilderImpl builder) {
        this.name = builder.name;
        this.successValues = builder.successValues;
        this.failureValues = builder.failureValues;
    }

    /**
     * <p>
     * The name of the header to match against. The name must be an exact match, including case.
     * </p>
     * <p>
     * JSON example: <code>"Name": [ "RequestResult" ]</code>
     * </p>
     * 
     * @return The name of the header to match against. The name must be an exact match, including case.</p>
     *         <p>
     *         JSON example: <code>"Name": [ "RequestResult" ]</code>
     */
    public final String name() {
        return name;
    }

    /**
     * For responses, this returns true if the service returned a value for the SuccessValues property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasSuccessValues() {
        return successValues != null && !(successValues instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Values in the response header with the specified name that indicate a successful login or account creation
     * attempt. To be counted as a success, the value must be an exact match, including case. Each value must be unique
     * among the success and failure values.
     * </p>
     * <p>
     * JSON examples: <code>"SuccessValues": [ "LoginPassed", "Successful login" ]</code> and
     * <code>"SuccessValues": [ "AccountCreated", "Successful account creation" ]</code>
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasSuccessValues} method.
     * </p>
     * 
     * @return Values in the response header with the specified name that indicate a successful login or account
     *         creation attempt. To be counted as a success, the value must be an exact match, including case. Each
     *         value must be unique among the success and failure values. </p>
     *         <p>
     *         JSON examples: <code>"SuccessValues": [ "LoginPassed", "Successful login" ]</code> and
     *         <code>"SuccessValues": [ "AccountCreated", "Successful account creation" ]</code>
     */
    public final List<String> successValues() {
        return successValues;
    }

    /**
     * For responses, this returns true if the service returned a value for the FailureValues property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasFailureValues() {
        return failureValues != null && !(failureValues instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Values in the response header with the specified name that indicate a failed login or account creation attempt.
     * To be counted as a failure, the value must be an exact match, including case. Each value must be unique among the
     * success and failure values.
     * </p>
     * <p>
     * JSON examples: <code>"FailureValues": [ "LoginFailed", "Failed login" ]</code> and
     * <code>"FailureValues": [ "AccountCreationFailed" ]</code>
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasFailureValues} method.
     * </p>
     * 
     * @return Values in the response header with the specified name that indicate a failed login or account creation
     *         attempt. To be counted as a failure, the value must be an exact match, including case. Each value must be
     *         unique among the success and failure values. </p>
     *         <p>
     *         JSON examples: <code>"FailureValues": [ "LoginFailed", "Failed login" ]</code> and
     *         <code>"FailureValues": [ "AccountCreationFailed" ]</code>
     */
    public final List<String> failureValues() {
        return failureValues;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(hasSuccessValues() ? successValues() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasFailureValues() ? failureValues() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ResponseInspectionHeader)) {
            return false;
        }
        ResponseInspectionHeader other = (ResponseInspectionHeader) obj;
        return Objects.equals(name(), other.name()) && hasSuccessValues() == other.hasSuccessValues()
                && Objects.equals(successValues(), other.successValues()) && hasFailureValues() == other.hasFailureValues()
                && Objects.equals(failureValues(), other.failureValues());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ResponseInspectionHeader").add("Name", name())
                .add("SuccessValues", hasSuccessValues() ? successValues() : null)
                .add("FailureValues", hasFailureValues() ? failureValues() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "SuccessValues":
            return Optional.ofNullable(clazz.cast(successValues()));
        case "FailureValues":
            return Optional.ofNullable(clazz.cast(failureValues()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("Name", NAME_FIELD);
        map.put("SuccessValues", SUCCESS_VALUES_FIELD);
        map.put("FailureValues", FAILURE_VALUES_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<ResponseInspectionHeader, T> g) {
        return obj -> g.apply((ResponseInspectionHeader) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ResponseInspectionHeader> {
        /**
         * <p>
         * The name of the header to match against. The name must be an exact match, including case.
         * </p>
         * <p>
         * JSON example: <code>"Name": [ "RequestResult" ]</code>
         * </p>
         * 
         * @param name
         *        The name of the header to match against. The name must be an exact match, including case.</p>
         *        <p>
         *        JSON example: <code>"Name": [ "RequestResult" ]</code>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * Values in the response header with the specified name that indicate a successful login or account creation
         * attempt. To be counted as a success, the value must be an exact match, including case. Each value must be
         * unique among the success and failure values.
         * </p>
         * <p>
         * JSON examples: <code>"SuccessValues": [ "LoginPassed", "Successful login" ]</code> and
         * <code>"SuccessValues": [ "AccountCreated", "Successful account creation" ]</code>
         * </p>
         * 
         * @param successValues
         *        Values in the response header with the specified name that indicate a successful login or account
         *        creation attempt. To be counted as a success, the value must be an exact match, including case. Each
         *        value must be unique among the success and failure values. </p>
         *        <p>
         *        JSON examples: <code>"SuccessValues": [ "LoginPassed", "Successful login" ]</code> and
         *        <code>"SuccessValues": [ "AccountCreated", "Successful account creation" ]</code>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder successValues(Collection<String> successValues);

        /**
         * <p>
         * Values in the response header with the specified name that indicate a successful login or account creation
         * attempt. To be counted as a success, the value must be an exact match, including case. Each value must be
         * unique among the success and failure values.
         * </p>
         * <p>
         * JSON examples: <code>"SuccessValues": [ "LoginPassed", "Successful login" ]</code> and
         * <code>"SuccessValues": [ "AccountCreated", "Successful account creation" ]</code>
         * </p>
         * 
         * @param successValues
         *        Values in the response header with the specified name that indicate a successful login or account
         *        creation attempt. To be counted as a success, the value must be an exact match, including case. Each
         *        value must be unique among the success and failure values. </p>
         *        <p>
         *        JSON examples: <code>"SuccessValues": [ "LoginPassed", "Successful login" ]</code> and
         *        <code>"SuccessValues": [ "AccountCreated", "Successful account creation" ]</code>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder successValues(String... successValues);

        /**
         * <p>
         * Values in the response header with the specified name that indicate a failed login or account creation
         * attempt. To be counted as a failure, the value must be an exact match, including case. Each value must be
         * unique among the success and failure values.
         * </p>
         * <p>
         * JSON examples: <code>"FailureValues": [ "LoginFailed", "Failed login" ]</code> and
         * <code>"FailureValues": [ "AccountCreationFailed" ]</code>
         * </p>
         * 
         * @param failureValues
         *        Values in the response header with the specified name that indicate a failed login or account creation
         *        attempt. To be counted as a failure, the value must be an exact match, including case. Each value must
         *        be unique among the success and failure values. </p>
         *        <p>
         *        JSON examples: <code>"FailureValues": [ "LoginFailed", "Failed login" ]</code> and
         *        <code>"FailureValues": [ "AccountCreationFailed" ]</code>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder failureValues(Collection<String> failureValues);

        /**
         * <p>
         * Values in the response header with the specified name that indicate a failed login or account creation
         * attempt. To be counted as a failure, the value must be an exact match, including case. Each value must be
         * unique among the success and failure values.
         * </p>
         * <p>
         * JSON examples: <code>"FailureValues": [ "LoginFailed", "Failed login" ]</code> and
         * <code>"FailureValues": [ "AccountCreationFailed" ]</code>
         * </p>
         * 
         * @param failureValues
         *        Values in the response header with the specified name that indicate a failed login or account creation
         *        attempt. To be counted as a failure, the value must be an exact match, including case. Each value must
         *        be unique among the success and failure values. </p>
         *        <p>
         *        JSON examples: <code>"FailureValues": [ "LoginFailed", "Failed login" ]</code> and
         *        <code>"FailureValues": [ "AccountCreationFailed" ]</code>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder failureValues(String... failureValues);
    }

    static final class BuilderImpl implements Builder {
        private String name;

        private List<String> successValues = DefaultSdkAutoConstructList.getInstance();

        private List<String> failureValues = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(ResponseInspectionHeader model) {
            name(model.name);
            successValues(model.successValues);
            failureValues(model.failureValues);
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final Collection<String> getSuccessValues() {
            if (successValues instanceof SdkAutoConstructList) {
                return null;
            }
            return successValues;
        }

        public final void setSuccessValues(Collection<String> successValues) {
            this.successValues = ResponseInspectionHeaderSuccessValuesCopier.copy(successValues);
        }

        @Override
        public final Builder successValues(Collection<String> successValues) {
            this.successValues = ResponseInspectionHeaderSuccessValuesCopier.copy(successValues);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder successValues(String... successValues) {
            successValues(Arrays.asList(successValues));
            return this;
        }

        public final Collection<String> getFailureValues() {
            if (failureValues instanceof SdkAutoConstructList) {
                return null;
            }
            return failureValues;
        }

        public final void setFailureValues(Collection<String> failureValues) {
            this.failureValues = ResponseInspectionHeaderFailureValuesCopier.copy(failureValues);
        }

        @Override
        public final Builder failureValues(Collection<String> failureValues) {
            this.failureValues = ResponseInspectionHeaderFailureValuesCopier.copy(failureValues);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder failureValues(String... failureValues) {
            failureValues(Arrays.asList(failureValues));
            return this;
        }

        @Override
        public ResponseInspectionHeader build() {
            return new ResponseInspectionHeader(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
