/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.lambda.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Specific configuration settings for a Kafka schema registry.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class KafkaSchemaRegistryConfig implements SdkPojo, Serializable,
        ToCopyableBuilder<KafkaSchemaRegistryConfig.Builder, KafkaSchemaRegistryConfig> {
    private static final SdkField<String> SCHEMA_REGISTRY_URI_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SchemaRegistryURI").getter(getter(KafkaSchemaRegistryConfig::schemaRegistryURI))
            .setter(setter(Builder::schemaRegistryURI))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SchemaRegistryURI").build()).build();

    private static final SdkField<String> EVENT_RECORD_FORMAT_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("EventRecordFormat").getter(getter(KafkaSchemaRegistryConfig::eventRecordFormatAsString))
            .setter(setter(Builder::eventRecordFormat))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EventRecordFormat").build()).build();

    private static final SdkField<List<KafkaSchemaRegistryAccessConfig>> ACCESS_CONFIGS_FIELD = SdkField
            .<List<KafkaSchemaRegistryAccessConfig>> builder(MarshallingType.LIST)
            .memberName("AccessConfigs")
            .getter(getter(KafkaSchemaRegistryConfig::accessConfigs))
            .setter(setter(Builder::accessConfigs))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AccessConfigs").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<KafkaSchemaRegistryAccessConfig> builder(MarshallingType.SDK_POJO)
                                            .constructor(KafkaSchemaRegistryAccessConfig::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<KafkaSchemaValidationConfig>> SCHEMA_VALIDATION_CONFIGS_FIELD = SdkField
            .<List<KafkaSchemaValidationConfig>> builder(MarshallingType.LIST)
            .memberName("SchemaValidationConfigs")
            .getter(getter(KafkaSchemaRegistryConfig::schemaValidationConfigs))
            .setter(setter(Builder::schemaValidationConfigs))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SchemaValidationConfigs").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<KafkaSchemaValidationConfig> builder(MarshallingType.SDK_POJO)
                                            .constructor(KafkaSchemaValidationConfig::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(SCHEMA_REGISTRY_URI_FIELD,
            EVENT_RECORD_FORMAT_FIELD, ACCESS_CONFIGS_FIELD, SCHEMA_VALIDATION_CONFIGS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String schemaRegistryURI;

    private final String eventRecordFormat;

    private final List<KafkaSchemaRegistryAccessConfig> accessConfigs;

    private final List<KafkaSchemaValidationConfig> schemaValidationConfigs;

    private KafkaSchemaRegistryConfig(BuilderImpl builder) {
        this.schemaRegistryURI = builder.schemaRegistryURI;
        this.eventRecordFormat = builder.eventRecordFormat;
        this.accessConfigs = builder.accessConfigs;
        this.schemaValidationConfigs = builder.schemaValidationConfigs;
    }

    /**
     * <p>
     * The URI for your schema registry. The correct URI format depends on the type of schema registry you're using.
     * </p>
     * <ul>
     * <li>
     * <p>
     * For Glue schema registries, use the ARN of the registry.
     * </p>
     * </li>
     * <li>
     * <p>
     * For Confluent schema registries, use the URL of the registry.
     * </p>
     * </li>
     * </ul>
     * 
     * @return The URI for your schema registry. The correct URI format depends on the type of schema registry you're
     *         using.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         For Glue schema registries, use the ARN of the registry.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         For Confluent schema registries, use the URL of the registry.
     *         </p>
     *         </li>
     */
    public final String schemaRegistryURI() {
        return schemaRegistryURI;
    }

    /**
     * <p>
     * The record format that Lambda delivers to your function after schema validation.
     * </p>
     * <ul>
     * <li>
     * <p>
     * Choose <code>JSON</code> to have Lambda deliver the record to your function as a standard JSON object.
     * </p>
     * </li>
     * <li>
     * <p>
     * Choose <code>SOURCE</code> to have Lambda deliver the record to your function in its original source format.
     * Lambda removes all schema metadata, such as the schema ID, before sending the record to your function.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #eventRecordFormat}
     * will return {@link SchemaRegistryEventRecordFormat#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service
     * is available from {@link #eventRecordFormatAsString}.
     * </p>
     * 
     * @return The record format that Lambda delivers to your function after schema validation.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         Choose <code>JSON</code> to have Lambda deliver the record to your function as a standard JSON object.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Choose <code>SOURCE</code> to have Lambda deliver the record to your function in its original source
     *         format. Lambda removes all schema metadata, such as the schema ID, before sending the record to your
     *         function.
     *         </p>
     *         </li>
     * @see SchemaRegistryEventRecordFormat
     */
    public final SchemaRegistryEventRecordFormat eventRecordFormat() {
        return SchemaRegistryEventRecordFormat.fromValue(eventRecordFormat);
    }

    /**
     * <p>
     * The record format that Lambda delivers to your function after schema validation.
     * </p>
     * <ul>
     * <li>
     * <p>
     * Choose <code>JSON</code> to have Lambda deliver the record to your function as a standard JSON object.
     * </p>
     * </li>
     * <li>
     * <p>
     * Choose <code>SOURCE</code> to have Lambda deliver the record to your function in its original source format.
     * Lambda removes all schema metadata, such as the schema ID, before sending the record to your function.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #eventRecordFormat}
     * will return {@link SchemaRegistryEventRecordFormat#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service
     * is available from {@link #eventRecordFormatAsString}.
     * </p>
     * 
     * @return The record format that Lambda delivers to your function after schema validation.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         Choose <code>JSON</code> to have Lambda deliver the record to your function as a standard JSON object.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Choose <code>SOURCE</code> to have Lambda deliver the record to your function in its original source
     *         format. Lambda removes all schema metadata, such as the schema ID, before sending the record to your
     *         function.
     *         </p>
     *         </li>
     * @see SchemaRegistryEventRecordFormat
     */
    public final String eventRecordFormatAsString() {
        return eventRecordFormat;
    }

    /**
     * For responses, this returns true if the service returned a value for the AccessConfigs property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasAccessConfigs() {
        return accessConfigs != null && !(accessConfigs instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * An array of access configuration objects that tell Lambda how to authenticate with your schema registry.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasAccessConfigs} method.
     * </p>
     * 
     * @return An array of access configuration objects that tell Lambda how to authenticate with your schema registry.
     */
    public final List<KafkaSchemaRegistryAccessConfig> accessConfigs() {
        return accessConfigs;
    }

    /**
     * For responses, this returns true if the service returned a value for the SchemaValidationConfigs property. This
     * DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasSchemaValidationConfigs() {
        return schemaValidationConfigs != null && !(schemaValidationConfigs instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * An array of schema validation configuration objects, which tell Lambda the message attributes you want to
     * validate and filter using your schema registry.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasSchemaValidationConfigs} method.
     * </p>
     * 
     * @return An array of schema validation configuration objects, which tell Lambda the message attributes you want to
     *         validate and filter using your schema registry.
     */
    public final List<KafkaSchemaValidationConfig> schemaValidationConfigs() {
        return schemaValidationConfigs;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(schemaRegistryURI());
        hashCode = 31 * hashCode + Objects.hashCode(eventRecordFormatAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasAccessConfigs() ? accessConfigs() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasSchemaValidationConfigs() ? schemaValidationConfigs() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof KafkaSchemaRegistryConfig)) {
            return false;
        }
        KafkaSchemaRegistryConfig other = (KafkaSchemaRegistryConfig) obj;
        return Objects.equals(schemaRegistryURI(), other.schemaRegistryURI())
                && Objects.equals(eventRecordFormatAsString(), other.eventRecordFormatAsString())
                && hasAccessConfigs() == other.hasAccessConfigs() && Objects.equals(accessConfigs(), other.accessConfigs())
                && hasSchemaValidationConfigs() == other.hasSchemaValidationConfigs()
                && Objects.equals(schemaValidationConfigs(), other.schemaValidationConfigs());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("KafkaSchemaRegistryConfig").add("SchemaRegistryURI", schemaRegistryURI())
                .add("EventRecordFormat", eventRecordFormatAsString())
                .add("AccessConfigs", hasAccessConfigs() ? accessConfigs() : null)
                .add("SchemaValidationConfigs", hasSchemaValidationConfigs() ? schemaValidationConfigs() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "SchemaRegistryURI":
            return Optional.ofNullable(clazz.cast(schemaRegistryURI()));
        case "EventRecordFormat":
            return Optional.ofNullable(clazz.cast(eventRecordFormatAsString()));
        case "AccessConfigs":
            return Optional.ofNullable(clazz.cast(accessConfigs()));
        case "SchemaValidationConfigs":
            return Optional.ofNullable(clazz.cast(schemaValidationConfigs()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("SchemaRegistryURI", SCHEMA_REGISTRY_URI_FIELD);
        map.put("EventRecordFormat", EVENT_RECORD_FORMAT_FIELD);
        map.put("AccessConfigs", ACCESS_CONFIGS_FIELD);
        map.put("SchemaValidationConfigs", SCHEMA_VALIDATION_CONFIGS_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<KafkaSchemaRegistryConfig, T> g) {
        return obj -> g.apply((KafkaSchemaRegistryConfig) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, KafkaSchemaRegistryConfig> {
        /**
         * <p>
         * The URI for your schema registry. The correct URI format depends on the type of schema registry you're using.
         * </p>
         * <ul>
         * <li>
         * <p>
         * For Glue schema registries, use the ARN of the registry.
         * </p>
         * </li>
         * <li>
         * <p>
         * For Confluent schema registries, use the URL of the registry.
         * </p>
         * </li>
         * </ul>
         * 
         * @param schemaRegistryURI
         *        The URI for your schema registry. The correct URI format depends on the type of schema registry you're
         *        using.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        For Glue schema registries, use the ARN of the registry.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For Confluent schema registries, use the URL of the registry.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder schemaRegistryURI(String schemaRegistryURI);

        /**
         * <p>
         * The record format that Lambda delivers to your function after schema validation.
         * </p>
         * <ul>
         * <li>
         * <p>
         * Choose <code>JSON</code> to have Lambda deliver the record to your function as a standard JSON object.
         * </p>
         * </li>
         * <li>
         * <p>
         * Choose <code>SOURCE</code> to have Lambda deliver the record to your function in its original source format.
         * Lambda removes all schema metadata, such as the schema ID, before sending the record to your function.
         * </p>
         * </li>
         * </ul>
         * 
         * @param eventRecordFormat
         *        The record format that Lambda delivers to your function after schema validation.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        Choose <code>JSON</code> to have Lambda deliver the record to your function as a standard JSON object.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Choose <code>SOURCE</code> to have Lambda deliver the record to your function in its original source
         *        format. Lambda removes all schema metadata, such as the schema ID, before sending the record to your
         *        function.
         *        </p>
         *        </li>
         * @see SchemaRegistryEventRecordFormat
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see SchemaRegistryEventRecordFormat
         */
        Builder eventRecordFormat(String eventRecordFormat);

        /**
         * <p>
         * The record format that Lambda delivers to your function after schema validation.
         * </p>
         * <ul>
         * <li>
         * <p>
         * Choose <code>JSON</code> to have Lambda deliver the record to your function as a standard JSON object.
         * </p>
         * </li>
         * <li>
         * <p>
         * Choose <code>SOURCE</code> to have Lambda deliver the record to your function in its original source format.
         * Lambda removes all schema metadata, such as the schema ID, before sending the record to your function.
         * </p>
         * </li>
         * </ul>
         * 
         * @param eventRecordFormat
         *        The record format that Lambda delivers to your function after schema validation.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        Choose <code>JSON</code> to have Lambda deliver the record to your function as a standard JSON object.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Choose <code>SOURCE</code> to have Lambda deliver the record to your function in its original source
         *        format. Lambda removes all schema metadata, such as the schema ID, before sending the record to your
         *        function.
         *        </p>
         *        </li>
         * @see SchemaRegistryEventRecordFormat
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see SchemaRegistryEventRecordFormat
         */
        Builder eventRecordFormat(SchemaRegistryEventRecordFormat eventRecordFormat);

        /**
         * <p>
         * An array of access configuration objects that tell Lambda how to authenticate with your schema registry.
         * </p>
         * 
         * @param accessConfigs
         *        An array of access configuration objects that tell Lambda how to authenticate with your schema
         *        registry.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder accessConfigs(Collection<KafkaSchemaRegistryAccessConfig> accessConfigs);

        /**
         * <p>
         * An array of access configuration objects that tell Lambda how to authenticate with your schema registry.
         * </p>
         * 
         * @param accessConfigs
         *        An array of access configuration objects that tell Lambda how to authenticate with your schema
         *        registry.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder accessConfigs(KafkaSchemaRegistryAccessConfig... accessConfigs);

        /**
         * <p>
         * An array of access configuration objects that tell Lambda how to authenticate with your schema registry.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.lambda.model.KafkaSchemaRegistryAccessConfig.Builder} avoiding the
         * need to create one manually via
         * {@link software.amazon.awssdk.services.lambda.model.KafkaSchemaRegistryAccessConfig#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.lambda.model.KafkaSchemaRegistryAccessConfig.Builder#build()} is
         * called immediately and its result is passed to {@link #accessConfigs(List<KafkaSchemaRegistryAccessConfig>)}.
         * 
         * @param accessConfigs
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.lambda.model.KafkaSchemaRegistryAccessConfig.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #accessConfigs(java.util.Collection<KafkaSchemaRegistryAccessConfig>)
         */
        Builder accessConfigs(Consumer<KafkaSchemaRegistryAccessConfig.Builder>... accessConfigs);

        /**
         * <p>
         * An array of schema validation configuration objects, which tell Lambda the message attributes you want to
         * validate and filter using your schema registry.
         * </p>
         * 
         * @param schemaValidationConfigs
         *        An array of schema validation configuration objects, which tell Lambda the message attributes you want
         *        to validate and filter using your schema registry.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder schemaValidationConfigs(Collection<KafkaSchemaValidationConfig> schemaValidationConfigs);

        /**
         * <p>
         * An array of schema validation configuration objects, which tell Lambda the message attributes you want to
         * validate and filter using your schema registry.
         * </p>
         * 
         * @param schemaValidationConfigs
         *        An array of schema validation configuration objects, which tell Lambda the message attributes you want
         *        to validate and filter using your schema registry.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder schemaValidationConfigs(KafkaSchemaValidationConfig... schemaValidationConfigs);

        /**
         * <p>
         * An array of schema validation configuration objects, which tell Lambda the message attributes you want to
         * validate and filter using your schema registry.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.lambda.model.KafkaSchemaValidationConfig.Builder} avoiding the need to
         * create one manually via
         * {@link software.amazon.awssdk.services.lambda.model.KafkaSchemaValidationConfig#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.lambda.model.KafkaSchemaValidationConfig.Builder#build()} is called
         * immediately and its result is passed to {@link #schemaValidationConfigs(List<KafkaSchemaValidationConfig>)}.
         * 
         * @param schemaValidationConfigs
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.lambda.model.KafkaSchemaValidationConfig.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #schemaValidationConfigs(java.util.Collection<KafkaSchemaValidationConfig>)
         */
        Builder schemaValidationConfigs(Consumer<KafkaSchemaValidationConfig.Builder>... schemaValidationConfigs);
    }

    static final class BuilderImpl implements Builder {
        private String schemaRegistryURI;

        private String eventRecordFormat;

        private List<KafkaSchemaRegistryAccessConfig> accessConfigs = DefaultSdkAutoConstructList.getInstance();

        private List<KafkaSchemaValidationConfig> schemaValidationConfigs = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(KafkaSchemaRegistryConfig model) {
            schemaRegistryURI(model.schemaRegistryURI);
            eventRecordFormat(model.eventRecordFormat);
            accessConfigs(model.accessConfigs);
            schemaValidationConfigs(model.schemaValidationConfigs);
        }

        public final String getSchemaRegistryURI() {
            return schemaRegistryURI;
        }

        public final void setSchemaRegistryURI(String schemaRegistryURI) {
            this.schemaRegistryURI = schemaRegistryURI;
        }

        @Override
        public final Builder schemaRegistryURI(String schemaRegistryURI) {
            this.schemaRegistryURI = schemaRegistryURI;
            return this;
        }

        public final String getEventRecordFormat() {
            return eventRecordFormat;
        }

        public final void setEventRecordFormat(String eventRecordFormat) {
            this.eventRecordFormat = eventRecordFormat;
        }

        @Override
        public final Builder eventRecordFormat(String eventRecordFormat) {
            this.eventRecordFormat = eventRecordFormat;
            return this;
        }

        @Override
        public final Builder eventRecordFormat(SchemaRegistryEventRecordFormat eventRecordFormat) {
            this.eventRecordFormat(eventRecordFormat == null ? null : eventRecordFormat.toString());
            return this;
        }

        public final List<KafkaSchemaRegistryAccessConfig.Builder> getAccessConfigs() {
            List<KafkaSchemaRegistryAccessConfig.Builder> result = KafkaSchemaRegistryAccessConfigListCopier
                    .copyToBuilder(this.accessConfigs);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setAccessConfigs(Collection<KafkaSchemaRegistryAccessConfig.BuilderImpl> accessConfigs) {
            this.accessConfigs = KafkaSchemaRegistryAccessConfigListCopier.copyFromBuilder(accessConfigs);
        }

        @Override
        public final Builder accessConfigs(Collection<KafkaSchemaRegistryAccessConfig> accessConfigs) {
            this.accessConfigs = KafkaSchemaRegistryAccessConfigListCopier.copy(accessConfigs);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder accessConfigs(KafkaSchemaRegistryAccessConfig... accessConfigs) {
            accessConfigs(Arrays.asList(accessConfigs));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder accessConfigs(Consumer<KafkaSchemaRegistryAccessConfig.Builder>... accessConfigs) {
            accessConfigs(Stream.of(accessConfigs).map(c -> KafkaSchemaRegistryAccessConfig.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final List<KafkaSchemaValidationConfig.Builder> getSchemaValidationConfigs() {
            List<KafkaSchemaValidationConfig.Builder> result = KafkaSchemaValidationConfigListCopier
                    .copyToBuilder(this.schemaValidationConfigs);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setSchemaValidationConfigs(Collection<KafkaSchemaValidationConfig.BuilderImpl> schemaValidationConfigs) {
            this.schemaValidationConfigs = KafkaSchemaValidationConfigListCopier.copyFromBuilder(schemaValidationConfigs);
        }

        @Override
        public final Builder schemaValidationConfigs(Collection<KafkaSchemaValidationConfig> schemaValidationConfigs) {
            this.schemaValidationConfigs = KafkaSchemaValidationConfigListCopier.copy(schemaValidationConfigs);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder schemaValidationConfigs(KafkaSchemaValidationConfig... schemaValidationConfigs) {
            schemaValidationConfigs(Arrays.asList(schemaValidationConfigs));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder schemaValidationConfigs(Consumer<KafkaSchemaValidationConfig.Builder>... schemaValidationConfigs) {
            schemaValidationConfigs(Stream.of(schemaValidationConfigs)
                    .map(c -> KafkaSchemaValidationConfig.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        @Override
        public KafkaSchemaRegistryConfig build() {
            return new KafkaSchemaRegistryConfig(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
