/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.lambda.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Specific access configuration settings that tell Lambda how to authenticate with your schema registry.
 * </p>
 * <p>
 * If you're working with an Glue schema registry, don't provide authentication details in this object. Instead, ensure
 * that your execution role has the required permissions for Lambda to access your cluster.
 * </p>
 * <p>
 * If you're working with a Confluent schema registry, choose the authentication method in the <code>Type</code> field,
 * and provide the Secrets Manager secret ARN in the <code>URI</code> field.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class KafkaSchemaRegistryAccessConfig implements SdkPojo, Serializable,
        ToCopyableBuilder<KafkaSchemaRegistryAccessConfig.Builder, KafkaSchemaRegistryAccessConfig> {
    private static final SdkField<String> TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Type")
            .getter(getter(KafkaSchemaRegistryAccessConfig::typeAsString)).setter(setter(Builder::type))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Type").build()).build();

    private static final SdkField<String> URI_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("URI")
            .getter(getter(KafkaSchemaRegistryAccessConfig::uri)).setter(setter(Builder::uri))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("URI").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(TYPE_FIELD, URI_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String type;

    private final String uri;

    private KafkaSchemaRegistryAccessConfig(BuilderImpl builder) {
        this.type = builder.type;
        this.uri = builder.uri;
    }

    /**
     * <p>
     * The type of authentication Lambda uses to access your schema registry.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link KafkaSchemaRegistryAuthType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #typeAsString}.
     * </p>
     * 
     * @return The type of authentication Lambda uses to access your schema registry.
     * @see KafkaSchemaRegistryAuthType
     */
    public final KafkaSchemaRegistryAuthType type() {
        return KafkaSchemaRegistryAuthType.fromValue(type);
    }

    /**
     * <p>
     * The type of authentication Lambda uses to access your schema registry.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link KafkaSchemaRegistryAuthType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #typeAsString}.
     * </p>
     * 
     * @return The type of authentication Lambda uses to access your schema registry.
     * @see KafkaSchemaRegistryAuthType
     */
    public final String typeAsString() {
        return type;
    }

    /**
     * <p>
     * The URI of the secret (Secrets Manager secret ARN) to authenticate with your schema registry.
     * </p>
     * 
     * @return The URI of the secret (Secrets Manager secret ARN) to authenticate with your schema registry.
     */
    public final String uri() {
        return uri;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(typeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(uri());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof KafkaSchemaRegistryAccessConfig)) {
            return false;
        }
        KafkaSchemaRegistryAccessConfig other = (KafkaSchemaRegistryAccessConfig) obj;
        return Objects.equals(typeAsString(), other.typeAsString()) && Objects.equals(uri(), other.uri());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("KafkaSchemaRegistryAccessConfig").add("Type", typeAsString()).add("URI", uri()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Type":
            return Optional.ofNullable(clazz.cast(typeAsString()));
        case "URI":
            return Optional.ofNullable(clazz.cast(uri()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("Type", TYPE_FIELD);
        map.put("URI", URI_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<KafkaSchemaRegistryAccessConfig, T> g) {
        return obj -> g.apply((KafkaSchemaRegistryAccessConfig) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, KafkaSchemaRegistryAccessConfig> {
        /**
         * <p>
         * The type of authentication Lambda uses to access your schema registry.
         * </p>
         * 
         * @param type
         *        The type of authentication Lambda uses to access your schema registry.
         * @see KafkaSchemaRegistryAuthType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see KafkaSchemaRegistryAuthType
         */
        Builder type(String type);

        /**
         * <p>
         * The type of authentication Lambda uses to access your schema registry.
         * </p>
         * 
         * @param type
         *        The type of authentication Lambda uses to access your schema registry.
         * @see KafkaSchemaRegistryAuthType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see KafkaSchemaRegistryAuthType
         */
        Builder type(KafkaSchemaRegistryAuthType type);

        /**
         * <p>
         * The URI of the secret (Secrets Manager secret ARN) to authenticate with your schema registry.
         * </p>
         * 
         * @param uri
         *        The URI of the secret (Secrets Manager secret ARN) to authenticate with your schema registry.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder uri(String uri);
    }

    static final class BuilderImpl implements Builder {
        private String type;

        private String uri;

        private BuilderImpl() {
        }

        private BuilderImpl(KafkaSchemaRegistryAccessConfig model) {
            type(model.type);
            uri(model.uri);
        }

        public final String getType() {
            return type;
        }

        public final void setType(String type) {
            this.type = type;
        }

        @Override
        public final Builder type(String type) {
            this.type = type;
            return this;
        }

        @Override
        public final Builder type(KafkaSchemaRegistryAuthType type) {
            this.type(type == null ? null : type.toString());
            return this;
        }

        public final String getUri() {
            return uri;
        }

        public final void setUri(String uri) {
            this.uri = uri;
        }

        @Override
        public final Builder uri(String uri) {
            this.uri = uri;
            return this;
        }

        @Override
        public KafkaSchemaRegistryAccessConfig build() {
            return new KafkaSchemaRegistryAccessConfig(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
