/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.iam.model;

import java.io.Serializable;
import java.nio.ByteBuffer;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkBytes;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains information about a virtual MFA device.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class VirtualMFADevice implements SdkPojo, Serializable,
        ToCopyableBuilder<VirtualMFADevice.Builder, VirtualMFADevice> {
    private static final SdkField<String> SERIAL_NUMBER_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SerialNumber").getter(getter(VirtualMFADevice::serialNumber)).setter(setter(Builder::serialNumber))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SerialNumber").build()).build();

    private static final SdkField<SdkBytes> BASE32_STRING_SEED_FIELD = SdkField.<SdkBytes> builder(MarshallingType.SDK_BYTES)
            .memberName("Base32StringSeed").getter(getter(VirtualMFADevice::base32StringSeed))
            .setter(setter(Builder::base32StringSeed))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Base32StringSeed").build()).build();

    private static final SdkField<SdkBytes> QR_CODE_PNG_FIELD = SdkField.<SdkBytes> builder(MarshallingType.SDK_BYTES)
            .memberName("QRCodePNG").getter(getter(VirtualMFADevice::qrCodePNG)).setter(setter(Builder::qrCodePNG))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("QRCodePNG").build()).build();

    private static final SdkField<User> USER_FIELD = SdkField.<User> builder(MarshallingType.SDK_POJO).memberName("User")
            .getter(getter(VirtualMFADevice::user)).setter(setter(Builder::user)).constructor(User::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("User").build()).build();

    private static final SdkField<Instant> ENABLE_DATE_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("EnableDate").getter(getter(VirtualMFADevice::enableDate)).setter(setter(Builder::enableDate))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EnableDate").build()).build();

    private static final SdkField<List<Tag>> TAGS_FIELD = SdkField
            .<List<Tag>> builder(MarshallingType.LIST)
            .memberName("Tags")
            .getter(getter(VirtualMFADevice::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Tags").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Tag> builder(MarshallingType.SDK_POJO)
                                            .constructor(Tag::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(SERIAL_NUMBER_FIELD,
            BASE32_STRING_SEED_FIELD, QR_CODE_PNG_FIELD, USER_FIELD, ENABLE_DATE_FIELD, TAGS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String serialNumber;

    private final SdkBytes base32StringSeed;

    private final SdkBytes qrCodePNG;

    private final User user;

    private final Instant enableDate;

    private final List<Tag> tags;

    private VirtualMFADevice(BuilderImpl builder) {
        this.serialNumber = builder.serialNumber;
        this.base32StringSeed = builder.base32StringSeed;
        this.qrCodePNG = builder.qrCodePNG;
        this.user = builder.user;
        this.enableDate = builder.enableDate;
        this.tags = builder.tags;
    }

    /**
     * <p>
     * The serial number associated with <code>VirtualMFADevice</code>.
     * </p>
     * 
     * @return The serial number associated with <code>VirtualMFADevice</code>.
     */
    public final String serialNumber() {
        return serialNumber;
    }

    /**
     * <p>
     * The base32 seed defined as specified in <a href="https://tools.ietf.org/html/rfc3548.txt">RFC3548</a>. The
     * <code>Base32StringSeed</code> is base32-encoded.
     * </p>
     * 
     * @return The base32 seed defined as specified in <a href="https://tools.ietf.org/html/rfc3548.txt">RFC3548</a>.
     *         The <code>Base32StringSeed</code> is base32-encoded.
     */
    public final SdkBytes base32StringSeed() {
        return base32StringSeed;
    }

    /**
     * <p>
     * A QR code PNG image that encodes
     * <code>otpauth://totp/$virtualMFADeviceName@$AccountName?secret=$Base32String</code> where
     * <code>$virtualMFADeviceName</code> is one of the create call arguments. <code>AccountName</code> is the user name
     * if set (otherwise, the account ID otherwise), and <code>Base32String</code> is the seed in base32 format. The
     * <code>Base32String</code> value is base64-encoded.
     * </p>
     * 
     * @return A QR code PNG image that encodes
     *         <code>otpauth://totp/$virtualMFADeviceName@$AccountName?secret=$Base32String</code> where
     *         <code>$virtualMFADeviceName</code> is one of the create call arguments. <code>AccountName</code> is the
     *         user name if set (otherwise, the account ID otherwise), and <code>Base32String</code> is the seed in
     *         base32 format. The <code>Base32String</code> value is base64-encoded.
     */
    public final SdkBytes qrCodePNG() {
        return qrCodePNG;
    }

    /**
     * <p>
     * The IAM user associated with this virtual MFA device.
     * </p>
     * 
     * @return The IAM user associated with this virtual MFA device.
     */
    public final User user() {
        return user;
    }

    /**
     * <p>
     * The date and time on which the virtual MFA device was enabled.
     * </p>
     * 
     * @return The date and time on which the virtual MFA device was enabled.
     */
    public final Instant enableDate() {
        return enableDate;
    }

    /**
     * For responses, this returns true if the service returned a value for the Tags property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of tags that are attached to the virtual MFA device. For more information about tagging, see <a
     * href="https://docs.aws.amazon.com/IAM/latest/UserGuide/id_tags.html">Tagging IAM resources</a> in the <i>IAM User
     * Guide</i>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTags} method.
     * </p>
     * 
     * @return A list of tags that are attached to the virtual MFA device. For more information about tagging, see <a
     *         href="https://docs.aws.amazon.com/IAM/latest/UserGuide/id_tags.html">Tagging IAM resources</a> in the
     *         <i>IAM User Guide</i>.
     */
    public final List<Tag> tags() {
        return tags;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(serialNumber());
        hashCode = 31 * hashCode + Objects.hashCode(base32StringSeed());
        hashCode = 31 * hashCode + Objects.hashCode(qrCodePNG());
        hashCode = 31 * hashCode + Objects.hashCode(user());
        hashCode = 31 * hashCode + Objects.hashCode(enableDate());
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof VirtualMFADevice)) {
            return false;
        }
        VirtualMFADevice other = (VirtualMFADevice) obj;
        return Objects.equals(serialNumber(), other.serialNumber())
                && Objects.equals(base32StringSeed(), other.base32StringSeed()) && Objects.equals(qrCodePNG(), other.qrCodePNG())
                && Objects.equals(user(), other.user()) && Objects.equals(enableDate(), other.enableDate())
                && hasTags() == other.hasTags() && Objects.equals(tags(), other.tags());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("VirtualMFADevice").add("SerialNumber", serialNumber())
                .add("Base32StringSeed", base32StringSeed() == null ? null : "*** Sensitive Data Redacted ***")
                .add("QRCodePNG", qrCodePNG() == null ? null : "*** Sensitive Data Redacted ***").add("User", user())
                .add("EnableDate", enableDate()).add("Tags", hasTags() ? tags() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "SerialNumber":
            return Optional.ofNullable(clazz.cast(serialNumber()));
        case "Base32StringSeed":
            return Optional.ofNullable(clazz.cast(base32StringSeed()));
        case "QRCodePNG":
            return Optional.ofNullable(clazz.cast(qrCodePNG()));
        case "User":
            return Optional.ofNullable(clazz.cast(user()));
        case "EnableDate":
            return Optional.ofNullable(clazz.cast(enableDate()));
        case "Tags":
            return Optional.ofNullable(clazz.cast(tags()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("SerialNumber", SERIAL_NUMBER_FIELD);
        map.put("Base32StringSeed", BASE32_STRING_SEED_FIELD);
        map.put("QRCodePNG", QR_CODE_PNG_FIELD);
        map.put("User", USER_FIELD);
        map.put("EnableDate", ENABLE_DATE_FIELD);
        map.put("Tags", TAGS_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<VirtualMFADevice, T> g) {
        return obj -> g.apply((VirtualMFADevice) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, VirtualMFADevice> {
        /**
         * <p>
         * The serial number associated with <code>VirtualMFADevice</code>.
         * </p>
         * 
         * @param serialNumber
         *        The serial number associated with <code>VirtualMFADevice</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder serialNumber(String serialNumber);

        /**
         * <p>
         * The base32 seed defined as specified in <a href="https://tools.ietf.org/html/rfc3548.txt">RFC3548</a>. The
         * <code>Base32StringSeed</code> is base32-encoded.
         * </p>
         * 
         * @param base32StringSeed
         *        The base32 seed defined as specified in <a href="https://tools.ietf.org/html/rfc3548.txt">RFC3548</a>.
         *        The <code>Base32StringSeed</code> is base32-encoded.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder base32StringSeed(SdkBytes base32StringSeed);

        /**
         * <p>
         * A QR code PNG image that encodes
         * <code>otpauth://totp/$virtualMFADeviceName@$AccountName?secret=$Base32String</code> where
         * <code>$virtualMFADeviceName</code> is one of the create call arguments. <code>AccountName</code> is the user
         * name if set (otherwise, the account ID otherwise), and <code>Base32String</code> is the seed in base32
         * format. The <code>Base32String</code> value is base64-encoded.
         * </p>
         * 
         * @param qrCodePNG
         *        A QR code PNG image that encodes
         *        <code>otpauth://totp/$virtualMFADeviceName@$AccountName?secret=$Base32String</code> where
         *        <code>$virtualMFADeviceName</code> is one of the create call arguments. <code>AccountName</code> is
         *        the user name if set (otherwise, the account ID otherwise), and <code>Base32String</code> is the seed
         *        in base32 format. The <code>Base32String</code> value is base64-encoded.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder qrCodePNG(SdkBytes qrCodePNG);

        /**
         * <p>
         * The IAM user associated with this virtual MFA device.
         * </p>
         * 
         * @param user
         *        The IAM user associated with this virtual MFA device.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder user(User user);

        /**
         * <p>
         * The IAM user associated with this virtual MFA device.
         * </p>
         * This is a convenience method that creates an instance of the {@link User.Builder} avoiding the need to create
         * one manually via {@link User#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link User.Builder#build()} is called immediately and its result is
         * passed to {@link #user(User)}.
         * 
         * @param user
         *        a consumer that will call methods on {@link User.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #user(User)
         */
        default Builder user(Consumer<User.Builder> user) {
            return user(User.builder().applyMutation(user).build());
        }

        /**
         * <p>
         * The date and time on which the virtual MFA device was enabled.
         * </p>
         * 
         * @param enableDate
         *        The date and time on which the virtual MFA device was enabled.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder enableDate(Instant enableDate);

        /**
         * <p>
         * A list of tags that are attached to the virtual MFA device. For more information about tagging, see <a
         * href="https://docs.aws.amazon.com/IAM/latest/UserGuide/id_tags.html">Tagging IAM resources</a> in the <i>IAM
         * User Guide</i>.
         * </p>
         * 
         * @param tags
         *        A list of tags that are attached to the virtual MFA device. For more information about tagging, see <a
         *        href="https://docs.aws.amazon.com/IAM/latest/UserGuide/id_tags.html">Tagging IAM resources</a> in the
         *        <i>IAM User Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Collection<Tag> tags);

        /**
         * <p>
         * A list of tags that are attached to the virtual MFA device. For more information about tagging, see <a
         * href="https://docs.aws.amazon.com/IAM/latest/UserGuide/id_tags.html">Tagging IAM resources</a> in the <i>IAM
         * User Guide</i>.
         * </p>
         * 
         * @param tags
         *        A list of tags that are attached to the virtual MFA device. For more information about tagging, see <a
         *        href="https://docs.aws.amazon.com/IAM/latest/UserGuide/id_tags.html">Tagging IAM resources</a> in the
         *        <i>IAM User Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Tag... tags);

        /**
         * <p>
         * A list of tags that are attached to the virtual MFA device. For more information about tagging, see <a
         * href="https://docs.aws.amazon.com/IAM/latest/UserGuide/id_tags.html">Tagging IAM resources</a> in the <i>IAM
         * User Guide</i>.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.iam.model.Tag.Builder} avoiding the need to create one manually via
         * {@link software.amazon.awssdk.services.iam.model.Tag#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link software.amazon.awssdk.services.iam.model.Tag.Builder#build()} is
         * called immediately and its result is passed to {@link #tags(List<Tag>)}.
         * 
         * @param tags
         *        a consumer that will call methods on {@link software.amazon.awssdk.services.iam.model.Tag.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tags(java.util.Collection<Tag>)
         */
        Builder tags(Consumer<Tag.Builder>... tags);
    }

    static final class BuilderImpl implements Builder {
        private String serialNumber;

        private SdkBytes base32StringSeed;

        private SdkBytes qrCodePNG;

        private User user;

        private Instant enableDate;

        private List<Tag> tags = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(VirtualMFADevice model) {
            serialNumber(model.serialNumber);
            base32StringSeed(model.base32StringSeed);
            qrCodePNG(model.qrCodePNG);
            user(model.user);
            enableDate(model.enableDate);
            tags(model.tags);
        }

        public final String getSerialNumber() {
            return serialNumber;
        }

        public final void setSerialNumber(String serialNumber) {
            this.serialNumber = serialNumber;
        }

        @Override
        public final Builder serialNumber(String serialNumber) {
            this.serialNumber = serialNumber;
            return this;
        }

        public final ByteBuffer getBase32StringSeed() {
            return base32StringSeed == null ? null : base32StringSeed.asByteBuffer();
        }

        public final void setBase32StringSeed(ByteBuffer base32StringSeed) {
            base32StringSeed(base32StringSeed == null ? null : SdkBytes.fromByteBuffer(base32StringSeed));
        }

        @Override
        public final Builder base32StringSeed(SdkBytes base32StringSeed) {
            this.base32StringSeed = base32StringSeed;
            return this;
        }

        public final ByteBuffer getQrCodePNG() {
            return qrCodePNG == null ? null : qrCodePNG.asByteBuffer();
        }

        public final void setQrCodePNG(ByteBuffer qrCodePNG) {
            qrCodePNG(qrCodePNG == null ? null : SdkBytes.fromByteBuffer(qrCodePNG));
        }

        @Override
        public final Builder qrCodePNG(SdkBytes qrCodePNG) {
            this.qrCodePNG = qrCodePNG;
            return this;
        }

        public final User.Builder getUser() {
            return user != null ? user.toBuilder() : null;
        }

        public final void setUser(User.BuilderImpl user) {
            this.user = user != null ? user.build() : null;
        }

        @Override
        public final Builder user(User user) {
            this.user = user;
            return this;
        }

        public final Instant getEnableDate() {
            return enableDate;
        }

        public final void setEnableDate(Instant enableDate) {
            this.enableDate = enableDate;
        }

        @Override
        public final Builder enableDate(Instant enableDate) {
            this.enableDate = enableDate;
            return this;
        }

        public final List<Tag.Builder> getTags() {
            List<Tag.Builder> result = _tagListTypeCopier.copyToBuilder(this.tags);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setTags(Collection<Tag.BuilderImpl> tags) {
            this.tags = _tagListTypeCopier.copyFromBuilder(tags);
        }

        @Override
        public final Builder tags(Collection<Tag> tags) {
            this.tags = _tagListTypeCopier.copy(tags);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Tag... tags) {
            tags(Arrays.asList(tags));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Consumer<Tag.Builder>... tags) {
            tags(Stream.of(tags).map(c -> Tag.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        @Override
        public VirtualMFADevice build() {
            return new VirtualMFADevice(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
