/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.wafv2.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A set of rules that is managed by Amazon Web Services and Amazon Web Services Marketplace sellers to provide
 * versioned managed rule groups for customers of WAF.
 * </p>
 * <note>
 * <p>
 * This is intended for use only by vendors of managed rule sets. Vendors are Amazon Web Services and Amazon Web
 * Services Marketplace sellers.
 * </p>
 * <p>
 * Vendors, you can use the managed rule set APIs to provide controlled rollout of your versioned managed rule group
 * offerings for your customers. The APIs are <code>ListManagedRuleSets</code>, <code>GetManagedRuleSet</code>,
 * <code>PutManagedRuleSetVersions</code>, and <code>UpdateManagedRuleSetVersionExpiryDate</code>.
 * </p>
 * </note>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ManagedRuleSet implements SdkPojo, Serializable, ToCopyableBuilder<ManagedRuleSet.Builder, ManagedRuleSet> {
    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Name")
            .getter(getter(ManagedRuleSet::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name").build()).build();

    private static final SdkField<String> ID_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Id")
            .getter(getter(ManagedRuleSet::id)).setter(setter(Builder::id))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Id").build()).build();

    private static final SdkField<String> ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("ARN")
            .getter(getter(ManagedRuleSet::arn)).setter(setter(Builder::arn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ARN").build()).build();

    private static final SdkField<String> DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Description").getter(getter(ManagedRuleSet::description)).setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Description").build()).build();

    private static final SdkField<Map<String, ManagedRuleSetVersion>> PUBLISHED_VERSIONS_FIELD = SdkField
            .<Map<String, ManagedRuleSetVersion>> builder(MarshallingType.MAP)
            .memberName("PublishedVersions")
            .getter(getter(ManagedRuleSet::publishedVersions))
            .setter(setter(Builder::publishedVersions))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PublishedVersions").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<ManagedRuleSetVersion> builder(MarshallingType.SDK_POJO)
                                            .constructor(ManagedRuleSetVersion::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<String> RECOMMENDED_VERSION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("RecommendedVersion").getter(getter(ManagedRuleSet::recommendedVersion))
            .setter(setter(Builder::recommendedVersion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RecommendedVersion").build())
            .build();

    private static final SdkField<String> LABEL_NAMESPACE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("LabelNamespace").getter(getter(ManagedRuleSet::labelNamespace)).setter(setter(Builder::labelNamespace))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LabelNamespace").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAME_FIELD, ID_FIELD,
            ARN_FIELD, DESCRIPTION_FIELD, PUBLISHED_VERSIONS_FIELD, RECOMMENDED_VERSION_FIELD, LABEL_NAMESPACE_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String name;

    private final String id;

    private final String arn;

    private final String description;

    private final Map<String, ManagedRuleSetVersion> publishedVersions;

    private final String recommendedVersion;

    private final String labelNamespace;

    private ManagedRuleSet(BuilderImpl builder) {
        this.name = builder.name;
        this.id = builder.id;
        this.arn = builder.arn;
        this.description = builder.description;
        this.publishedVersions = builder.publishedVersions;
        this.recommendedVersion = builder.recommendedVersion;
        this.labelNamespace = builder.labelNamespace;
    }

    /**
     * <p>
     * The name of the managed rule set. You use this, along with the rule set ID, to identify the rule set.
     * </p>
     * <p>
     * This name is assigned to the corresponding managed rule group, which your customers can access and use.
     * </p>
     * 
     * @return The name of the managed rule set. You use this, along with the rule set ID, to identify the rule set.</p>
     *         <p>
     *         This name is assigned to the corresponding managed rule group, which your customers can access and use.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * A unique identifier for the managed rule set. The ID is returned in the responses to commands like
     * <code>list</code>. You provide it to operations like <code>get</code> and <code>update</code>.
     * </p>
     * 
     * @return A unique identifier for the managed rule set. The ID is returned in the responses to commands like
     *         <code>list</code>. You provide it to operations like <code>get</code> and <code>update</code>.
     */
    public final String id() {
        return id;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the entity.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the entity.
     */
    public final String arn() {
        return arn;
    }

    /**
     * <p>
     * A description of the set that helps with identification.
     * </p>
     * 
     * @return A description of the set that helps with identification.
     */
    public final String description() {
        return description;
    }

    /**
     * For responses, this returns true if the service returned a value for the PublishedVersions property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasPublishedVersions() {
        return publishedVersions != null && !(publishedVersions instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * The versions of this managed rule set that are available for use by customers.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasPublishedVersions} method.
     * </p>
     * 
     * @return The versions of this managed rule set that are available for use by customers.
     */
    public final Map<String, ManagedRuleSetVersion> publishedVersions() {
        return publishedVersions;
    }

    /**
     * <p>
     * The version that you would like your customers to use.
     * </p>
     * 
     * @return The version that you would like your customers to use.
     */
    public final String recommendedVersion() {
        return recommendedVersion;
    }

    /**
     * <p>
     * The label namespace prefix for the managed rule groups that are offered to customers from this managed rule set.
     * All labels that are added by rules in the managed rule group have this prefix.
     * </p>
     * <ul>
     * <li>
     * <p>
     * The syntax for the label namespace prefix for a managed rule group is the following:
     * </p>
     * <p>
     * <code>awswaf:managed:&lt;vendor&gt;:&lt;rule group name&gt;</code>:
     * </p>
     * </li>
     * <li>
     * <p>
     * When a rule with a label matches a web request, WAF adds the fully qualified label to the request. A fully
     * qualified label is made up of the label namespace from the rule group or web ACL where the rule is defined and
     * the label from the rule, separated by a colon:
     * </p>
     * <p>
     * <code>&lt;label namespace&gt;:&lt;label from rule&gt;</code>
     * </p>
     * </li>
     * </ul>
     * 
     * @return The label namespace prefix for the managed rule groups that are offered to customers from this managed
     *         rule set. All labels that are added by rules in the managed rule group have this prefix. </p>
     *         <ul>
     *         <li>
     *         <p>
     *         The syntax for the label namespace prefix for a managed rule group is the following:
     *         </p>
     *         <p>
     *         <code>awswaf:managed:&lt;vendor&gt;:&lt;rule group name&gt;</code>:
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         When a rule with a label matches a web request, WAF adds the fully qualified label to the request. A
     *         fully qualified label is made up of the label namespace from the rule group or web ACL where the rule is
     *         defined and the label from the rule, separated by a colon:
     *         </p>
     *         <p>
     *         <code>&lt;label namespace&gt;:&lt;label from rule&gt;</code>
     *         </p>
     *         </li>
     */
    public final String labelNamespace() {
        return labelNamespace;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(id());
        hashCode = 31 * hashCode + Objects.hashCode(arn());
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(hasPublishedVersions() ? publishedVersions() : null);
        hashCode = 31 * hashCode + Objects.hashCode(recommendedVersion());
        hashCode = 31 * hashCode + Objects.hashCode(labelNamespace());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ManagedRuleSet)) {
            return false;
        }
        ManagedRuleSet other = (ManagedRuleSet) obj;
        return Objects.equals(name(), other.name()) && Objects.equals(id(), other.id()) && Objects.equals(arn(), other.arn())
                && Objects.equals(description(), other.description()) && hasPublishedVersions() == other.hasPublishedVersions()
                && Objects.equals(publishedVersions(), other.publishedVersions())
                && Objects.equals(recommendedVersion(), other.recommendedVersion())
                && Objects.equals(labelNamespace(), other.labelNamespace());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ManagedRuleSet").add("Name", name()).add("Id", id()).add("ARN", arn())
                .add("Description", description()).add("PublishedVersions", hasPublishedVersions() ? publishedVersions() : null)
                .add("RecommendedVersion", recommendedVersion()).add("LabelNamespace", labelNamespace()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "Id":
            return Optional.ofNullable(clazz.cast(id()));
        case "ARN":
            return Optional.ofNullable(clazz.cast(arn()));
        case "Description":
            return Optional.ofNullable(clazz.cast(description()));
        case "PublishedVersions":
            return Optional.ofNullable(clazz.cast(publishedVersions()));
        case "RecommendedVersion":
            return Optional.ofNullable(clazz.cast(recommendedVersion()));
        case "LabelNamespace":
            return Optional.ofNullable(clazz.cast(labelNamespace()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("Name", NAME_FIELD);
        map.put("Id", ID_FIELD);
        map.put("ARN", ARN_FIELD);
        map.put("Description", DESCRIPTION_FIELD);
        map.put("PublishedVersions", PUBLISHED_VERSIONS_FIELD);
        map.put("RecommendedVersion", RECOMMENDED_VERSION_FIELD);
        map.put("LabelNamespace", LABEL_NAMESPACE_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<ManagedRuleSet, T> g) {
        return obj -> g.apply((ManagedRuleSet) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ManagedRuleSet> {
        /**
         * <p>
         * The name of the managed rule set. You use this, along with the rule set ID, to identify the rule set.
         * </p>
         * <p>
         * This name is assigned to the corresponding managed rule group, which your customers can access and use.
         * </p>
         * 
         * @param name
         *        The name of the managed rule set. You use this, along with the rule set ID, to identify the rule
         *        set.</p>
         *        <p>
         *        This name is assigned to the corresponding managed rule group, which your customers can access and
         *        use.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * A unique identifier for the managed rule set. The ID is returned in the responses to commands like
         * <code>list</code>. You provide it to operations like <code>get</code> and <code>update</code>.
         * </p>
         * 
         * @param id
         *        A unique identifier for the managed rule set. The ID is returned in the responses to commands like
         *        <code>list</code>. You provide it to operations like <code>get</code> and <code>update</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder id(String id);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the entity.
         * </p>
         * 
         * @param arn
         *        The Amazon Resource Name (ARN) of the entity.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder arn(String arn);

        /**
         * <p>
         * A description of the set that helps with identification.
         * </p>
         * 
         * @param description
         *        A description of the set that helps with identification.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * The versions of this managed rule set that are available for use by customers.
         * </p>
         * 
         * @param publishedVersions
         *        The versions of this managed rule set that are available for use by customers.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder publishedVersions(Map<String, ManagedRuleSetVersion> publishedVersions);

        /**
         * <p>
         * The version that you would like your customers to use.
         * </p>
         * 
         * @param recommendedVersion
         *        The version that you would like your customers to use.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder recommendedVersion(String recommendedVersion);

        /**
         * <p>
         * The label namespace prefix for the managed rule groups that are offered to customers from this managed rule
         * set. All labels that are added by rules in the managed rule group have this prefix.
         * </p>
         * <ul>
         * <li>
         * <p>
         * The syntax for the label namespace prefix for a managed rule group is the following:
         * </p>
         * <p>
         * <code>awswaf:managed:&lt;vendor&gt;:&lt;rule group name&gt;</code>:
         * </p>
         * </li>
         * <li>
         * <p>
         * When a rule with a label matches a web request, WAF adds the fully qualified label to the request. A fully
         * qualified label is made up of the label namespace from the rule group or web ACL where the rule is defined
         * and the label from the rule, separated by a colon:
         * </p>
         * <p>
         * <code>&lt;label namespace&gt;:&lt;label from rule&gt;</code>
         * </p>
         * </li>
         * </ul>
         * 
         * @param labelNamespace
         *        The label namespace prefix for the managed rule groups that are offered to customers from this managed
         *        rule set. All labels that are added by rules in the managed rule group have this prefix. </p>
         *        <ul>
         *        <li>
         *        <p>
         *        The syntax for the label namespace prefix for a managed rule group is the following:
         *        </p>
         *        <p>
         *        <code>awswaf:managed:&lt;vendor&gt;:&lt;rule group name&gt;</code>:
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        When a rule with a label matches a web request, WAF adds the fully qualified label to the request. A
         *        fully qualified label is made up of the label namespace from the rule group or web ACL where the rule
         *        is defined and the label from the rule, separated by a colon:
         *        </p>
         *        <p>
         *        <code>&lt;label namespace&gt;:&lt;label from rule&gt;</code>
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder labelNamespace(String labelNamespace);
    }

    static final class BuilderImpl implements Builder {
        private String name;

        private String id;

        private String arn;

        private String description;

        private Map<String, ManagedRuleSetVersion> publishedVersions = DefaultSdkAutoConstructMap.getInstance();

        private String recommendedVersion;

        private String labelNamespace;

        private BuilderImpl() {
        }

        private BuilderImpl(ManagedRuleSet model) {
            name(model.name);
            id(model.id);
            arn(model.arn);
            description(model.description);
            publishedVersions(model.publishedVersions);
            recommendedVersion(model.recommendedVersion);
            labelNamespace(model.labelNamespace);
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final String getId() {
            return id;
        }

        public final void setId(String id) {
            this.id = id;
        }

        @Override
        public final Builder id(String id) {
            this.id = id;
            return this;
        }

        public final String getArn() {
            return arn;
        }

        public final void setArn(String arn) {
            this.arn = arn;
        }

        @Override
        public final Builder arn(String arn) {
            this.arn = arn;
            return this;
        }

        public final String getDescription() {
            return description;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final Map<String, ManagedRuleSetVersion.Builder> getPublishedVersions() {
            Map<String, ManagedRuleSetVersion.Builder> result = PublishedVersionsCopier.copyToBuilder(this.publishedVersions);
            if (result instanceof SdkAutoConstructMap) {
                return null;
            }
            return result;
        }

        public final void setPublishedVersions(Map<String, ManagedRuleSetVersion.BuilderImpl> publishedVersions) {
            this.publishedVersions = PublishedVersionsCopier.copyFromBuilder(publishedVersions);
        }

        @Override
        public final Builder publishedVersions(Map<String, ManagedRuleSetVersion> publishedVersions) {
            this.publishedVersions = PublishedVersionsCopier.copy(publishedVersions);
            return this;
        }

        public final String getRecommendedVersion() {
            return recommendedVersion;
        }

        public final void setRecommendedVersion(String recommendedVersion) {
            this.recommendedVersion = recommendedVersion;
        }

        @Override
        public final Builder recommendedVersion(String recommendedVersion) {
            this.recommendedVersion = recommendedVersion;
            return this;
        }

        public final String getLabelNamespace() {
            return labelNamespace;
        }

        public final void setLabelNamespace(String labelNamespace) {
            this.labelNamespace = labelNamespace;
        }

        @Override
        public final Builder labelNamespace(String labelNamespace) {
            this.labelNamespace = labelNamespace;
            return this;
        }

        @Override
        public ManagedRuleSet build() {
            return new ManagedRuleSet(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
