/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ec2.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Reserve powerful GPU instances on a future date to support your short duration machine learning (ML) workloads.
 * Instances that run inside a Capacity Block are automatically placed close together inside <a
 * href="http://aws.amazon.com/ec2/ultraclusters/">Amazon EC2 UltraClusters</a>, for low-latency, petabit-scale,
 * non-blocking networking.
 * </p>
 * <p>
 * You can also reserve Amazon EC2 UltraServers. UltraServers connect multiple EC2 instances using a low-latency,
 * high-bandwidth accelerator interconnect (NeuronLink). They are built to tackle very large-scale AI/ML workloads that
 * require significant processing power. For more information, see Amazon EC2 UltraServers.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class CapacityBlock implements SdkPojo, Serializable, ToCopyableBuilder<CapacityBlock.Builder, CapacityBlock> {
    private static final SdkField<String> CAPACITY_BLOCK_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("CapacityBlockId")
            .getter(getter(CapacityBlock::capacityBlockId))
            .setter(setter(Builder::capacityBlockId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CapacityBlockId")
                    .unmarshallLocationName("capacityBlockId").build()).build();

    private static final SdkField<String> ULTRASERVER_TYPE_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("UltraserverType")
            .getter(getter(CapacityBlock::ultraserverType))
            .setter(setter(Builder::ultraserverType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("UltraserverType")
                    .unmarshallLocationName("ultraserverType").build()).build();

    private static final SdkField<String> AVAILABILITY_ZONE_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("AvailabilityZone")
            .getter(getter(CapacityBlock::availabilityZone))
            .setter(setter(Builder::availabilityZone))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AvailabilityZone")
                    .unmarshallLocationName("availabilityZone").build()).build();

    private static final SdkField<String> AVAILABILITY_ZONE_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("AvailabilityZoneId")
            .getter(getter(CapacityBlock::availabilityZoneId))
            .setter(setter(Builder::availabilityZoneId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AvailabilityZoneId")
                    .unmarshallLocationName("availabilityZoneId").build()).build();

    private static final SdkField<List<String>> CAPACITY_RESERVATION_IDS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("CapacityReservationIds")
            .getter(getter(CapacityBlock::capacityReservationIds))
            .setter(setter(Builder::capacityReservationIds))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CapacityReservationIdSet")
                    .unmarshallLocationName("capacityReservationIdSet").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("item")
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("Item").unmarshallLocationName("item").build()).build())
                            .build()).build();

    private static final SdkField<Instant> START_DATE_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("StartDate")
            .getter(getter(CapacityBlock::startDate))
            .setter(setter(Builder::startDate))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StartDate")
                    .unmarshallLocationName("startDate").build()).build();

    private static final SdkField<Instant> END_DATE_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("EndDate")
            .getter(getter(CapacityBlock::endDate))
            .setter(setter(Builder::endDate))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EndDate")
                    .unmarshallLocationName("endDate").build()).build();

    private static final SdkField<Instant> CREATE_DATE_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("CreateDate")
            .getter(getter(CapacityBlock::createDate))
            .setter(setter(Builder::createDate))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreateDate")
                    .unmarshallLocationName("createDate").build()).build();

    private static final SdkField<String> STATE_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("State")
            .getter(getter(CapacityBlock::stateAsString))
            .setter(setter(Builder::state))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("State")
                    .unmarshallLocationName("state").build()).build();

    private static final SdkField<List<Tag>> TAGS_FIELD = SdkField
            .<List<Tag>> builder(MarshallingType.LIST)
            .memberName("Tags")
            .getter(getter(CapacityBlock::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TagSet")
                    .unmarshallLocationName("tagSet").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("item")
                            .memberFieldInfo(
                                    SdkField.<Tag> builder(MarshallingType.SDK_POJO)
                                            .constructor(Tag::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("Item").unmarshallLocationName("item").build()).build())
                            .build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(CAPACITY_BLOCK_ID_FIELD,
            ULTRASERVER_TYPE_FIELD, AVAILABILITY_ZONE_FIELD, AVAILABILITY_ZONE_ID_FIELD, CAPACITY_RESERVATION_IDS_FIELD,
            START_DATE_FIELD, END_DATE_FIELD, CREATE_DATE_FIELD, STATE_FIELD, TAGS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String capacityBlockId;

    private final String ultraserverType;

    private final String availabilityZone;

    private final String availabilityZoneId;

    private final List<String> capacityReservationIds;

    private final Instant startDate;

    private final Instant endDate;

    private final Instant createDate;

    private final String state;

    private final List<Tag> tags;

    private CapacityBlock(BuilderImpl builder) {
        this.capacityBlockId = builder.capacityBlockId;
        this.ultraserverType = builder.ultraserverType;
        this.availabilityZone = builder.availabilityZone;
        this.availabilityZoneId = builder.availabilityZoneId;
        this.capacityReservationIds = builder.capacityReservationIds;
        this.startDate = builder.startDate;
        this.endDate = builder.endDate;
        this.createDate = builder.createDate;
        this.state = builder.state;
        this.tags = builder.tags;
    }

    /**
     * <p>
     * The ID of the Capacity Block.
     * </p>
     * 
     * @return The ID of the Capacity Block.
     */
    public final String capacityBlockId() {
        return capacityBlockId;
    }

    /**
     * <p>
     * The EC2 UltraServer type of the Capacity Block.
     * </p>
     * 
     * @return The EC2 UltraServer type of the Capacity Block.
     */
    public final String ultraserverType() {
        return ultraserverType;
    }

    /**
     * <p>
     * The Availability Zone of the Capacity Block.
     * </p>
     * 
     * @return The Availability Zone of the Capacity Block.
     */
    public final String availabilityZone() {
        return availabilityZone;
    }

    /**
     * <p>
     * The Availability Zone ID of the Capacity Block.
     * </p>
     * 
     * @return The Availability Zone ID of the Capacity Block.
     */
    public final String availabilityZoneId() {
        return availabilityZoneId;
    }

    /**
     * For responses, this returns true if the service returned a value for the CapacityReservationIds property. This
     * DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasCapacityReservationIds() {
        return capacityReservationIds != null && !(capacityReservationIds instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The ID of the Capacity Reservation.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasCapacityReservationIds} method.
     * </p>
     * 
     * @return The ID of the Capacity Reservation.
     */
    public final List<String> capacityReservationIds() {
        return capacityReservationIds;
    }

    /**
     * <p>
     * The date and time at which the Capacity Block was started.
     * </p>
     * 
     * @return The date and time at which the Capacity Block was started.
     */
    public final Instant startDate() {
        return startDate;
    }

    /**
     * <p>
     * The date and time at which the Capacity Block expires. When a Capacity Block expires, all instances in the
     * Capacity Block are terminated.
     * </p>
     * 
     * @return The date and time at which the Capacity Block expires. When a Capacity Block expires, all instances in
     *         the Capacity Block are terminated.
     */
    public final Instant endDate() {
        return endDate;
    }

    /**
     * <p>
     * The date and time at which the Capacity Block was created.
     * </p>
     * 
     * @return The date and time at which the Capacity Block was created.
     */
    public final Instant createDate() {
        return createDate;
    }

    /**
     * <p>
     * The state of the Capacity Block.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #state} will return
     * {@link CapacityBlockResourceState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #stateAsString}.
     * </p>
     * 
     * @return The state of the Capacity Block.
     * @see CapacityBlockResourceState
     */
    public final CapacityBlockResourceState state() {
        return CapacityBlockResourceState.fromValue(state);
    }

    /**
     * <p>
     * The state of the Capacity Block.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #state} will return
     * {@link CapacityBlockResourceState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #stateAsString}.
     * </p>
     * 
     * @return The state of the Capacity Block.
     * @see CapacityBlockResourceState
     */
    public final String stateAsString() {
        return state;
    }

    /**
     * For responses, this returns true if the service returned a value for the Tags property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The tags assigned to the Capacity Block.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTags} method.
     * </p>
     * 
     * @return The tags assigned to the Capacity Block.
     */
    public final List<Tag> tags() {
        return tags;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(capacityBlockId());
        hashCode = 31 * hashCode + Objects.hashCode(ultraserverType());
        hashCode = 31 * hashCode + Objects.hashCode(availabilityZone());
        hashCode = 31 * hashCode + Objects.hashCode(availabilityZoneId());
        hashCode = 31 * hashCode + Objects.hashCode(hasCapacityReservationIds() ? capacityReservationIds() : null);
        hashCode = 31 * hashCode + Objects.hashCode(startDate());
        hashCode = 31 * hashCode + Objects.hashCode(endDate());
        hashCode = 31 * hashCode + Objects.hashCode(createDate());
        hashCode = 31 * hashCode + Objects.hashCode(stateAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CapacityBlock)) {
            return false;
        }
        CapacityBlock other = (CapacityBlock) obj;
        return Objects.equals(capacityBlockId(), other.capacityBlockId())
                && Objects.equals(ultraserverType(), other.ultraserverType())
                && Objects.equals(availabilityZone(), other.availabilityZone())
                && Objects.equals(availabilityZoneId(), other.availabilityZoneId())
                && hasCapacityReservationIds() == other.hasCapacityReservationIds()
                && Objects.equals(capacityReservationIds(), other.capacityReservationIds())
                && Objects.equals(startDate(), other.startDate()) && Objects.equals(endDate(), other.endDate())
                && Objects.equals(createDate(), other.createDate()) && Objects.equals(stateAsString(), other.stateAsString())
                && hasTags() == other.hasTags() && Objects.equals(tags(), other.tags());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("CapacityBlock").add("CapacityBlockId", capacityBlockId())
                .add("UltraserverType", ultraserverType()).add("AvailabilityZone", availabilityZone())
                .add("AvailabilityZoneId", availabilityZoneId())
                .add("CapacityReservationIds", hasCapacityReservationIds() ? capacityReservationIds() : null)
                .add("StartDate", startDate()).add("EndDate", endDate()).add("CreateDate", createDate())
                .add("State", stateAsString()).add("Tags", hasTags() ? tags() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "CapacityBlockId":
            return Optional.ofNullable(clazz.cast(capacityBlockId()));
        case "UltraserverType":
            return Optional.ofNullable(clazz.cast(ultraserverType()));
        case "AvailabilityZone":
            return Optional.ofNullable(clazz.cast(availabilityZone()));
        case "AvailabilityZoneId":
            return Optional.ofNullable(clazz.cast(availabilityZoneId()));
        case "CapacityReservationIds":
            return Optional.ofNullable(clazz.cast(capacityReservationIds()));
        case "StartDate":
            return Optional.ofNullable(clazz.cast(startDate()));
        case "EndDate":
            return Optional.ofNullable(clazz.cast(endDate()));
        case "CreateDate":
            return Optional.ofNullable(clazz.cast(createDate()));
        case "State":
            return Optional.ofNullable(clazz.cast(stateAsString()));
        case "Tags":
            return Optional.ofNullable(clazz.cast(tags()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("CapacityBlockId", CAPACITY_BLOCK_ID_FIELD);
        map.put("UltraserverType", ULTRASERVER_TYPE_FIELD);
        map.put("AvailabilityZone", AVAILABILITY_ZONE_FIELD);
        map.put("AvailabilityZoneId", AVAILABILITY_ZONE_ID_FIELD);
        map.put("CapacityReservationIdSet", CAPACITY_RESERVATION_IDS_FIELD);
        map.put("StartDate", START_DATE_FIELD);
        map.put("EndDate", END_DATE_FIELD);
        map.put("CreateDate", CREATE_DATE_FIELD);
        map.put("State", STATE_FIELD);
        map.put("TagSet", TAGS_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<CapacityBlock, T> g) {
        return obj -> g.apply((CapacityBlock) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, CapacityBlock> {
        /**
         * <p>
         * The ID of the Capacity Block.
         * </p>
         * 
         * @param capacityBlockId
         *        The ID of the Capacity Block.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder capacityBlockId(String capacityBlockId);

        /**
         * <p>
         * The EC2 UltraServer type of the Capacity Block.
         * </p>
         * 
         * @param ultraserverType
         *        The EC2 UltraServer type of the Capacity Block.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ultraserverType(String ultraserverType);

        /**
         * <p>
         * The Availability Zone of the Capacity Block.
         * </p>
         * 
         * @param availabilityZone
         *        The Availability Zone of the Capacity Block.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder availabilityZone(String availabilityZone);

        /**
         * <p>
         * The Availability Zone ID of the Capacity Block.
         * </p>
         * 
         * @param availabilityZoneId
         *        The Availability Zone ID of the Capacity Block.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder availabilityZoneId(String availabilityZoneId);

        /**
         * <p>
         * The ID of the Capacity Reservation.
         * </p>
         * 
         * @param capacityReservationIds
         *        The ID of the Capacity Reservation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder capacityReservationIds(Collection<String> capacityReservationIds);

        /**
         * <p>
         * The ID of the Capacity Reservation.
         * </p>
         * 
         * @param capacityReservationIds
         *        The ID of the Capacity Reservation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder capacityReservationIds(String... capacityReservationIds);

        /**
         * <p>
         * The date and time at which the Capacity Block was started.
         * </p>
         * 
         * @param startDate
         *        The date and time at which the Capacity Block was started.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder startDate(Instant startDate);

        /**
         * <p>
         * The date and time at which the Capacity Block expires. When a Capacity Block expires, all instances in the
         * Capacity Block are terminated.
         * </p>
         * 
         * @param endDate
         *        The date and time at which the Capacity Block expires. When a Capacity Block expires, all instances in
         *        the Capacity Block are terminated.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder endDate(Instant endDate);

        /**
         * <p>
         * The date and time at which the Capacity Block was created.
         * </p>
         * 
         * @param createDate
         *        The date and time at which the Capacity Block was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createDate(Instant createDate);

        /**
         * <p>
         * The state of the Capacity Block.
         * </p>
         * 
         * @param state
         *        The state of the Capacity Block.
         * @see CapacityBlockResourceState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see CapacityBlockResourceState
         */
        Builder state(String state);

        /**
         * <p>
         * The state of the Capacity Block.
         * </p>
         * 
         * @param state
         *        The state of the Capacity Block.
         * @see CapacityBlockResourceState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see CapacityBlockResourceState
         */
        Builder state(CapacityBlockResourceState state);

        /**
         * <p>
         * The tags assigned to the Capacity Block.
         * </p>
         * 
         * @param tags
         *        The tags assigned to the Capacity Block.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Collection<Tag> tags);

        /**
         * <p>
         * The tags assigned to the Capacity Block.
         * </p>
         * 
         * @param tags
         *        The tags assigned to the Capacity Block.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Tag... tags);

        /**
         * <p>
         * The tags assigned to the Capacity Block.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.ec2.model.Tag.Builder} avoiding the need to create one manually via
         * {@link software.amazon.awssdk.services.ec2.model.Tag#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link software.amazon.awssdk.services.ec2.model.Tag.Builder#build()} is
         * called immediately and its result is passed to {@link #tags(List<Tag>)}.
         * 
         * @param tags
         *        a consumer that will call methods on {@link software.amazon.awssdk.services.ec2.model.Tag.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tags(java.util.Collection<Tag>)
         */
        Builder tags(Consumer<Tag.Builder>... tags);
    }

    static final class BuilderImpl implements Builder {
        private String capacityBlockId;

        private String ultraserverType;

        private String availabilityZone;

        private String availabilityZoneId;

        private List<String> capacityReservationIds = DefaultSdkAutoConstructList.getInstance();

        private Instant startDate;

        private Instant endDate;

        private Instant createDate;

        private String state;

        private List<Tag> tags = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(CapacityBlock model) {
            capacityBlockId(model.capacityBlockId);
            ultraserverType(model.ultraserverType);
            availabilityZone(model.availabilityZone);
            availabilityZoneId(model.availabilityZoneId);
            capacityReservationIds(model.capacityReservationIds);
            startDate(model.startDate);
            endDate(model.endDate);
            createDate(model.createDate);
            state(model.state);
            tags(model.tags);
        }

        public final String getCapacityBlockId() {
            return capacityBlockId;
        }

        public final void setCapacityBlockId(String capacityBlockId) {
            this.capacityBlockId = capacityBlockId;
        }

        @Override
        public final Builder capacityBlockId(String capacityBlockId) {
            this.capacityBlockId = capacityBlockId;
            return this;
        }

        public final String getUltraserverType() {
            return ultraserverType;
        }

        public final void setUltraserverType(String ultraserverType) {
            this.ultraserverType = ultraserverType;
        }

        @Override
        public final Builder ultraserverType(String ultraserverType) {
            this.ultraserverType = ultraserverType;
            return this;
        }

        public final String getAvailabilityZone() {
            return availabilityZone;
        }

        public final void setAvailabilityZone(String availabilityZone) {
            this.availabilityZone = availabilityZone;
        }

        @Override
        public final Builder availabilityZone(String availabilityZone) {
            this.availabilityZone = availabilityZone;
            return this;
        }

        public final String getAvailabilityZoneId() {
            return availabilityZoneId;
        }

        public final void setAvailabilityZoneId(String availabilityZoneId) {
            this.availabilityZoneId = availabilityZoneId;
        }

        @Override
        public final Builder availabilityZoneId(String availabilityZoneId) {
            this.availabilityZoneId = availabilityZoneId;
            return this;
        }

        public final Collection<String> getCapacityReservationIds() {
            if (capacityReservationIds instanceof SdkAutoConstructList) {
                return null;
            }
            return capacityReservationIds;
        }

        public final void setCapacityReservationIds(Collection<String> capacityReservationIds) {
            this.capacityReservationIds = CapacityReservationIdSetCopier.copy(capacityReservationIds);
        }

        @Override
        public final Builder capacityReservationIds(Collection<String> capacityReservationIds) {
            this.capacityReservationIds = CapacityReservationIdSetCopier.copy(capacityReservationIds);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder capacityReservationIds(String... capacityReservationIds) {
            capacityReservationIds(Arrays.asList(capacityReservationIds));
            return this;
        }

        public final Instant getStartDate() {
            return startDate;
        }

        public final void setStartDate(Instant startDate) {
            this.startDate = startDate;
        }

        @Override
        public final Builder startDate(Instant startDate) {
            this.startDate = startDate;
            return this;
        }

        public final Instant getEndDate() {
            return endDate;
        }

        public final void setEndDate(Instant endDate) {
            this.endDate = endDate;
        }

        @Override
        public final Builder endDate(Instant endDate) {
            this.endDate = endDate;
            return this;
        }

        public final Instant getCreateDate() {
            return createDate;
        }

        public final void setCreateDate(Instant createDate) {
            this.createDate = createDate;
        }

        @Override
        public final Builder createDate(Instant createDate) {
            this.createDate = createDate;
            return this;
        }

        public final String getState() {
            return state;
        }

        public final void setState(String state) {
            this.state = state;
        }

        @Override
        public final Builder state(String state) {
            this.state = state;
            return this;
        }

        @Override
        public final Builder state(CapacityBlockResourceState state) {
            this.state(state == null ? null : state.toString());
            return this;
        }

        public final List<Tag.Builder> getTags() {
            List<Tag.Builder> result = TagListCopier.copyToBuilder(this.tags);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setTags(Collection<Tag.BuilderImpl> tags) {
            this.tags = TagListCopier.copyFromBuilder(tags);
        }

        @Override
        public final Builder tags(Collection<Tag> tags) {
            this.tags = TagListCopier.copy(tags);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Tag... tags) {
            tags(Arrays.asList(tags));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Consumer<Tag.Builder>... tags) {
            tags(Stream.of(tags).map(c -> Tag.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        @Override
        public CapacityBlock build() {
            return new CapacityBlock(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
