/*
 * Decompiled with CFR 0.152.
 */
package smile.regression;

import java.util.Arrays;
import smile.data.Attribute;
import smile.data.AttributeDataset;
import smile.data.NumericAttribute;
import smile.math.Math;
import smile.regression.Regression;
import smile.regression.RegressionTrainer;
import smile.regression.RegressionTree;
import smile.sort.QuickSelect;
import smile.util.SmileUtils;
import smile.validation.RMSE;
import smile.validation.RegressionMeasure;

public class GradientTreeBoost
implements Regression<double[]> {
    private static final long serialVersionUID = 1L;
    private RegressionTree[] trees;
    private double b = 0.0;
    private double[] importance;
    private Loss loss = Loss.LeastAbsoluteDeviation;
    private double shrinkage = 0.005;
    private int maxNodes = 6;
    private int ntrees = 500;
    private double f = 0.7;

    public GradientTreeBoost(double[][] x, double[] y, int ntrees) {
        this(null, x, y, ntrees);
    }

    public GradientTreeBoost(double[][] x, double[] y, Loss loss, int ntrees, int maxNodes, double shrinkage, double f) {
        this(null, x, y, loss, ntrees, maxNodes, shrinkage, f);
    }

    public GradientTreeBoost(Attribute[] attributes, double[][] x, double[] y, int ntrees) {
        this(attributes, x, y, Loss.LeastAbsoluteDeviation, ntrees, 6, x.length < 2000 ? 0.005 : 0.05, 0.7);
    }

    public GradientTreeBoost(AttributeDataset data, int ntrees) {
        this(data.attributes(), data.x(), data.y(), ntrees);
    }

    public GradientTreeBoost(AttributeDataset data, Loss loss, int ntrees, int maxNodes, double shrinkage, double f) {
        this(data.attributes(), data.x(), data.y(), loss, ntrees, maxNodes, shrinkage, f);
    }

    public GradientTreeBoost(Attribute[] attributes, double[][] x, double[] y, Loss loss, int ntrees, int maxNodes, double shrinkage, double f) {
        int i;
        if (x.length != y.length) {
            throw new IllegalArgumentException(String.format("The sizes of X and Y don't match: %d != %d", x.length, y.length));
        }
        if (shrinkage <= 0.0 || shrinkage > 1.0) {
            throw new IllegalArgumentException("Invalid shrinkage: " + shrinkage);
        }
        if (f <= 0.0 || f > 1.0) {
            throw new IllegalArgumentException("Invalid sampling fraction: " + f);
        }
        if (attributes == null) {
            int p = x[0].length;
            attributes = new Attribute[p];
            for (int i2 = 0; i2 < p; ++i2) {
                attributes[i2] = new NumericAttribute("V" + (i2 + 1));
            }
        }
        this.loss = loss;
        this.ntrees = ntrees;
        this.maxNodes = maxNodes;
        this.shrinkage = shrinkage;
        this.f = f;
        int n = x.length;
        int N = (int)Math.round((double)((double)n * f));
        int[] perm = new int[n];
        int[] samples = new int[n];
        for (int i3 = 0; i3 < n; ++i3) {
            perm[i3] = i3;
        }
        double[] residual = new double[n];
        double[] response = null;
        RegressionTree.NodeOutput output = null;
        if (loss == Loss.LeastSquares) {
            response = residual;
            this.b = Math.mean((double[])y);
            for (i = 0; i < n; ++i) {
                residual[i] = y[i] - this.b;
            }
        } else if (loss == Loss.LeastAbsoluteDeviation) {
            output = new LADNodeOutput(residual);
            System.arraycopy(y, 0, residual, 0, n);
            this.b = QuickSelect.median((double[])residual);
            response = new double[n];
            for (i = 0; i < n; ++i) {
                residual[i] = y[i] - this.b;
                response[i] = Math.signum((double)residual[i]);
            }
        } else if (loss == Loss.Huber) {
            response = new double[n];
            System.arraycopy(y, 0, residual, 0, n);
            this.b = QuickSelect.median((double[])residual);
            for (i = 0; i < n; ++i) {
                residual[i] = y[i] - this.b;
            }
        }
        int[][] order = SmileUtils.sort(attributes, x);
        this.trees = new RegressionTree[ntrees];
        for (int m = 0; m < ntrees; ++m) {
            int i4;
            Arrays.fill(samples, 0);
            Math.permutate((int[])perm);
            for (i4 = 0; i4 < N; ++i4) {
                samples[perm[i4]] = 1;
            }
            if (loss == Loss.Huber) {
                output = new HuberNodeOutput(residual, response, 0.9);
            }
            this.trees[m] = new RegressionTree(attributes, x, response, maxNodes, 5, x[0].length, order, samples, output);
            for (i4 = 0; i4 < n; ++i4) {
                int n2 = i4;
                residual[n2] = residual[n2] - shrinkage * this.trees[m].predict(x[i4]);
                if (loss != Loss.LeastAbsoluteDeviation) continue;
                response[i4] = Math.signum((double)residual[i4]);
            }
        }
        this.importance = new double[attributes.length];
        for (RegressionTree tree : this.trees) {
            double[] imp = tree.importance();
            for (int i5 = 0; i5 < imp.length; ++i5) {
                int n3 = i5;
                this.importance[n3] = this.importance[n3] + imp[i5];
            }
        }
    }

    public double[] importance() {
        return this.importance;
    }

    public double getSamplingRate() {
        return this.f;
    }

    public int getmaxNodes() {
        return this.maxNodes;
    }

    public Loss getLossFunction() {
        return this.loss;
    }

    public int size() {
        return this.trees.length;
    }

    public void trim(int ntrees) {
        if (ntrees > this.trees.length) {
            throw new IllegalArgumentException("The new model size is larger than the current size.");
        }
        if (ntrees < 1) {
            throw new IllegalArgumentException("Invalid new model size: " + ntrees);
        }
        if (ntrees < this.trees.length) {
            this.trees = Arrays.copyOf(this.trees, ntrees);
            this.ntrees = ntrees;
        }
    }

    @Override
    public double predict(double[] x) {
        double y = this.b;
        for (int i = 0; i < this.ntrees; ++i) {
            y += this.shrinkage * this.trees[i].predict(x);
        }
        return y;
    }

    public double[] test(double[][] x, double[] y) {
        double[] rmse = new double[this.ntrees];
        int n = x.length;
        double[] prediction = new double[n];
        Arrays.fill(prediction, this.b);
        RMSE measure = new RMSE();
        for (int i = 0; i < this.ntrees; ++i) {
            for (int j = 0; j < n; ++j) {
                int n2 = j;
                prediction[n2] = prediction[n2] + this.shrinkage * this.trees[i].predict(x[j]);
            }
            rmse[i] = measure.measure(y, prediction);
        }
        return rmse;
    }

    public double[][] test(double[][] x, double[] y, RegressionMeasure[] measures) {
        int m = measures.length;
        double[][] results = new double[this.ntrees][m];
        int n = x.length;
        double[] prediction = new double[n];
        Arrays.fill(prediction, this.b);
        for (int i = 0; i < this.ntrees; ++i) {
            int j;
            for (j = 0; j < n; ++j) {
                int n2 = j;
                prediction[n2] = prediction[n2] + this.shrinkage * this.trees[i].predict(x[j]);
            }
            for (j = 0; j < m; ++j) {
                results[i][j] = measures[j].measure(y, prediction);
            }
        }
        return results;
    }

    public RegressionTree[] getTrees() {
        return this.trees;
    }

    class HuberNodeOutput
    implements RegressionTree.NodeOutput {
        double[] residual;
        double[] response;
        double alpha;
        double delta;

        public HuberNodeOutput(double[] residual, double[] response, double alpha) {
            int i;
            this.residual = residual;
            this.response = response;
            this.alpha = alpha;
            int n = residual.length;
            for (i = 0; i < n; ++i) {
                response[i] = Math.abs((double)residual[i]);
            }
            this.delta = QuickSelect.select((double[])response, (int)((int)((double)n * alpha)));
            for (i = 0; i < n; ++i) {
                response[i] = Math.abs((double)residual[i]) <= this.delta ? residual[i] : this.delta * Math.signum((double)residual[i]);
            }
        }

        @Override
        public double calculate(int[] samples) {
            int n = 0;
            for (int s : samples) {
                if (s <= 0) continue;
                ++n;
            }
            double[] res = new double[n];
            int j = 0;
            for (int i = 0; i < samples.length; ++i) {
                if (samples[i] <= 0) continue;
                res[j++] = this.residual[i];
            }
            double r = QuickSelect.median((double[])res);
            double output = 0.0;
            for (int i = 0; i < samples.length; ++i) {
                if (samples[i] <= 0) continue;
                double d = this.residual[i] - r;
                output += Math.signum((double)d) * Math.min((double)this.delta, (double)Math.abs((double)d));
            }
            output = r + output / (double)n;
            return output;
        }
    }

    class LADNodeOutput
    implements RegressionTree.NodeOutput {
        double[] residual;

        public LADNodeOutput(double[] residual) {
            this.residual = residual;
        }

        @Override
        public double calculate(int[] samples) {
            int n = 0;
            for (int s : samples) {
                if (s <= 0) continue;
                ++n;
            }
            double[] r = new double[n];
            int j = 0;
            for (int i = 0; i < samples.length; ++i) {
                if (samples[i] <= 0) continue;
                r[j++] = this.residual[i];
            }
            return QuickSelect.median((double[])r);
        }
    }

    public static class Trainer
    extends RegressionTrainer<double[]> {
        private Loss loss = Loss.LeastAbsoluteDeviation;
        private int ntrees = 500;
        private double shrinkage = 0.005;
        private int maxNodes = 6;
        private double f = 0.7;

        public Trainer(int ntrees) {
            if (ntrees < 1) {
                throw new IllegalArgumentException("Invalid number of trees: " + ntrees);
            }
            this.ntrees = ntrees;
        }

        public Trainer(Attribute[] attributes, int ntrees) {
            super(attributes);
            if (ntrees < 1) {
                throw new IllegalArgumentException("Invalid number of trees: " + ntrees);
            }
            this.ntrees = ntrees;
        }

        public Trainer setLoss(Loss loss) {
            this.loss = loss;
            return this;
        }

        public Trainer setNumTrees(int ntrees) {
            if (ntrees < 1) {
                throw new IllegalArgumentException("Invalid number of trees: " + ntrees);
            }
            this.ntrees = ntrees;
            return this;
        }

        public Trainer setMaxNodes(int maxNodes) {
            if (maxNodes < 2) {
                throw new IllegalArgumentException("Invalid number of leaf nodes: " + maxNodes);
            }
            this.maxNodes = maxNodes;
            return this;
        }

        public Trainer setShrinkage(double shrinkage) {
            if (shrinkage <= 0.0 || shrinkage > 1.0) {
                throw new IllegalArgumentException("Invalid shrinkage: " + shrinkage);
            }
            this.shrinkage = shrinkage;
            return this;
        }

        public Trainer setSamplingRates(double f) {
            if (f <= 0.0 || f > 1.0) {
                throw new IllegalArgumentException("Invalid sampling fraction: " + f);
            }
            this.f = f;
            return this;
        }

        public GradientTreeBoost train(double[][] x, double[] y) {
            return new GradientTreeBoost(this.attributes, x, y, this.loss, this.ntrees, this.maxNodes, this.shrinkage, this.f);
        }
    }

    public static enum Loss {
        LeastSquares,
        LeastAbsoluteDeviation,
        Huber;

    }
}

